<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;
use App\Models\Sectorhidrico;
use App\Models\Precipitacionmensual;
use Illuminate\Support\Facades\Log;

class ActualizacionMensual extends Command
{
    protected $signature = 'actualizacion:mensual';

    protected $description = 'Realiza actualización mensual de datos desde el webservice de Ceazamet';

    public function __construct()
    {
        parent::__construct();
    }

    public function handle()
    {
        // Define el código del sensor de precipitación
        $sensorCode = 'CANPP';

        // Construye la URL de consulta para el sensor específico
        $url = "https://www.ceazamet.cl/ws/pop_ws.php?fn=GetSerieSensor&p_cod=ceazamet&s_cod={$sensorCode}&fecha_inicio=2013-01-01&fecha_fin=2013-12-31&interv=mes";

        try {
            // Realiza la consulta a la API de Ceazamet
            $response = Http::get($url);

            if ($response->successful()) {
                $responseData = $response->body();
                $lines = explode("\n", $responseData);

                // Ignoramos las primeras líneas de encabezado
                $lines = array_slice($lines, 5);

                foreach ($lines as $line) {
                    $data = str_getcsv($line, ',');

                    if (count($data) >= 6) { // Asegura que los datos necesarios estén presentes
                        $sensorCode = $data[0];
                        $ultimaLectura = $data[1];
                        $prom = $data[3];

                        // Extrae año y mes de la última lectura
                        $year = Carbon::parse($ultimaLectura)->year;
                        $month = Carbon::parse($ultimaLectura)->month;

                        // Encuentra los sectores hídricos relacionados con el sensor
                        $sectores = Sectorhidrico::whereHas('emaceazas', function ($query) use ($sensorCode) {
                            $query->where('sensorprecipitacion', $sensorCode);
                        })->get();

                        foreach ($sectores as $sector) {
                            $sectorId = $sector->id;

                            Log::info("Procesando datos para el sector hídrico $sectorId: Año $year, Mes $month, Promedio $prom");

                            // Inserta o actualiza el registro en la tabla 'precipitacionmensuals' con el sector relacionado
                            Precipitacionmensual::updateOrInsert([
                                'sectorhidrico_id' => $sectorId,
                                'anho' => $year,
                                'mes' => $month,
                            ], [
                                'pp' => $prom,
                            ]);

                            Log::info("Registro insertado/actualizado en precipitacionmensuals");
                        }
                    } else {
                        Log::warning("La respuesta de la API no contiene datos válidos");
                    }
                }
            } else {
                Log::warning("La respuesta de la API no pudo ser obtenida para el sensor CHILLPP");
            }
        } catch (\Exception $e) {
            // Maneja excepciones, como errores de solicitud a la API.
            // Registra errores o toma medidas específicas en caso de error.
            Log::error("Error al procesar la API de Ceazamet: " . $e->getMessage());
        }

        $this->info('Actualización mensual completada.');
    }
}
