<?php

namespace App\Http\Controllers;

use App\Models\Canal;
use App\Models\EficienciaRiegoSistema;
use App\Models\Kc;
use App\Models\OfertaSeccRioElquiCsiro;
use App\Models\OfertaSeccRioLimarCsiro;
use App\Models\OfertaSeccRioChoapaCsiro;
use App\Models\Etomensual;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class CalculoCanalCsiroController extends Controller
{
    public function calcularOfertaPorCanal($canal_id, $periodo, $superficie, $grupo_id, $sistema_riego, $daa, Request $request)
    {
        // Obtener canal con sus relaciones
        $canal = Canal::with(relations: ['sectorhidrico', 'seccionrio', 'comuna', 'subsubcuenca'])->find($canal_id);

        if (!$canal) {
            Log::error("Canal no encontrado para ID: $canal_id");
            return response()->json(['error' => 'Canal no encontrado'], 404);
        }

        // Obtener la eficiencia del sistema de riego
        $eficienciaSistema = EficienciaRiegoSistema::where('sistema', $sistema_riego)->first();
        if (!$eficienciaSistema) {
            Log::error("Eficiencia del sistema de riego no encontrada para sistema: $sistema_riego");
            return response()->json(['error' => 'Eficiencia del sistema de riego no encontrada'], 404);
        }

        // Validación de la eficiencia de conducción proporcionada por el usuario
        $eficienciaConduccion = $request->query('eficiencia_conduccion');
        if (!$eficienciaConduccion || $eficienciaConduccion < 1 || $eficienciaConduccion > 100) {
            Log::error("Eficiencia de conducción fuera de rango: $eficienciaConduccion%");
            return response()->json(['error' => 'Eficiencia de conducción debe estar entre 1% y 100%'], 400);
        }
        $eficienciaConduccion = $eficienciaConduccion / 100;

        // Obtener promedios de Kc y Eto
        $promedioKc = $this->calcularPromedioKc($grupo_id, $canal->sectorhidrico_id);
        $promedioEto = $this->calcularPromedioEto($canal->sectorhidrico_id);

        // Determinar la provincia y obtener el periodo de años para la temporada
        $provincia = $this->determinarProvincia($canal->comuna_id);
        list($startYear, $endYear) = $this->getSeasonPeriod($provincia, $periodo);

        // Inicializar las variables para oferta y demanda
        $rwsValues = [];
        $seasonalData = [];

        // Valores del pozo
        $demandaManual = $request->query('demanda_hidrica_manual');
        $demandaMensualManual = $demandaManual ? $demandaManual / 12 : null;

        // Iterar sobre cada año y cada mes para obtener la oferta y la demanda
        for ($year = $startYear; $year < $endYear; $year++) {
            $ofertaTotalTemporada = 0;
            $demandaTotalTemporada = 0;

            $seasonMonths = $this->calculateSeasonMonths($provincia, $year);
            foreach ($seasonMonths as $period) {
                $currentYear = $period['year'];
                $month = $period['month'];

                Log::info("Procesando datos para Año: $currentYear, Mes: $month");

                // Obtener valores de oferta, Kc y Eto
                $oferta = OfertaSeccRioElquiCsiro::where('seccionrio_id', $canal->seccionrio_id)
                    ->where('fecha', $currentYear)
                    ->where('mes', $month)
                    ->first()
                    ?? OfertaSeccRioLimarCsiro::where('seccionrio_id', $canal->seccionrio_id)
                    ->where('fecha', $currentYear)
                    ->where('mes', $month)
                    ->first()
                    ?? OfertaSeccRioChoapaCsiro::where('seccionrio_id', $canal->seccionrio_id)
                    ->where('fecha', $currentYear)
                    ->where('mes', $month)
                    ->first();

                if (!$oferta) {
                    Log::warning("Oferta no encontrada para Año: $currentYear, Mes: $month, Sección Río: {$canal->seccionrio_id}");
                }

                $kc = Kc::where('grupo_id', $grupo_id)
                    ->where('sectorhidrico_id', $canal->sectorhidrico_id)
                    ->where('anho', $currentYear)
                    ->where('mes', $month)
                    ->first();

                $eto = Etomensual::where('sectorhidrico_id', $canal->sectorhidrico_id)
                    ->where('anho', $currentYear)
                    ->where('mes', $month)
                    ->first();

                // Valores usados en el cálculo
                $kcValue = $kc->kc ?? $promedioKc;
                $etoValue = $eto->eto ?? $promedioEto;
                $eficienciaSistemaValue = max(0.1, $eficienciaSistema->eficiencia);

                Log::info("Valores obtenidos: Superficie: $superficie, Kc: $kcValue, Eto: $etoValue, Eficiencia Sistema: $eficienciaSistemaValue");

                if ($eficienciaSistemaValue == 0) {
                    Log::error("Eficiencia del sistema es 0, no se puede dividir.");
                    return response()->json(['error' => 'Eficiencia del sistema es inválida'], 500);
                }

                $demandaMensual = $demandaMensualManual
                    ? $demandaMensualManual
                    : ($superficie * $kcValue * $etoValue * 10) / $eficienciaSistemaValue;

                if ($demandaMensual == 0) {
                    Log::warning("Demanda mensual calculada como 0.");
                } else {
                    Log::info("Demanda mensual calculada: $demandaMensual");
                }

                $demandaTotalTemporada += $demandaMensual;
            
                // Calcular y registrar oferta mensual ajustada
                if ($oferta) {
                    $ofertaMensual = $oferta->ofertahidrica * $daa * $eficienciaConduccion;
                    Log::info("Oferta mensual ajustada calculada: $ofertaMensual (Oferta hídrica: {$oferta->ofertahidrica}, DAA: $daa, Eficiencia Conducción: $eficienciaConduccion)");
                    $ofertaTotalTemporada += $ofertaMensual;
                } else {
                    Log::warning("No se encontró oferta para Año $currentYear, Mes $month");
                }
            }
            
            // Log resumen para la temporada
            Log::info("Resumen Temporada $year: Demanda Total: $demandaTotalTemporada, Oferta Total: $ofertaTotalTemporada");
            
            // Solo agregar valores si la demanda es mayor a 0
            if ($demandaTotalTemporada > 0) {
                $rwsTemporada = $ofertaTotalTemporada / $demandaTotalTemporada;
            
                // Validar antes de agregar a los datos de la temporada
                if ($rwsTemporada == 0) {
                    Log::warning("RWS Temporada calculada como 0. Detalles: Oferta Total: $ofertaTotalTemporada, Demanda Total: $demandaTotalTemporada");
                } else {
                    Log::info("RWS Temporada calculada: $rwsTemporada");
                }
            
                // Guardar detalles de la temporada
                $seasonalData[] = [
                    'temporada' => "{$year}/" . ($year + 1),
                    'oferta' => $ofertaTotalTemporada,
                    'demanda' => $demandaTotalTemporada,
                    'rws' => $rwsTemporada,
                ];
            } else {
                Log::warning("Demanda total de la temporada es 0, no se puede calcular RWS. Oferta Total: $ofertaTotalTemporada");
            }
            
        }

        $seasonalDataWithDisponibilidad = [];
        foreach ($seasonalData as $season) {
            $disponibilidad = $this->calcularDisponibilidadPorSuperficie($season['oferta'], $superficie);
            $seasonalDataWithDisponibilidad[] = array_merge($season, [
                'disponibilidad_por_superficie' => $disponibilidad['disponibilidad'],
                'clasificacion_disponibilidad' => $disponibilidad['clasificacion'],
                'valor_clasificacion_disponibilidad' => $disponibilidad['valor_clasificacion'],
            ]);
        }

        // Agrupamos temporadas por periodo y calculamos probabilidad de excedencia
        $probabilidadExcedencia = $this->calcularProbabilidadExcedenciaWeibull($seasonalDataWithDisponibilidad);

        // Calcular percentiles de RWS
        $rwsValues = array_column($seasonalData, 'rws');
        rsort($rwsValues);
        $percentiles = $this->calculatePercentiles($rwsValues, [50, 75, 85, 90]);

        // Calcular la oferta en el percentil 85%
        $ofertaValues = array_column($seasonalData, 'oferta');
        rsort($ofertaValues);
        $oferta85 = $this->calculatePercentileValue($ofertaValues, 85);

        // Calcular la demanda hídrica promedio
        $demandaPromedio = array_sum(array_column($seasonalData, 'demanda')) / count($seasonalData);

        // Calcular las acciones extras necesarias para satisfacer la demanda usando la oferta85 y demandaPromedio
        $accionesExtras = $this->calcularAccionesExtras($demandaPromedio, $oferta85, $daa, $eficienciaConduccion);

        // Clasificar RWS en base a percentiles
        $rwsClassifications = [];
        foreach ($percentiles as $percentile => $rws) {
            $clasificacion = $this->clasificarRWS($rws);
            $rwsClassifications[$percentile] = [
                'rws' => $rws,
                'clasificacion' => $clasificacion,
            ];
        }
        

        return response()->json([
            'rws_values' => $rwsClassifications,
            'seasonal_data' => $seasonalDataWithDisponibilidad,
            'probabilidad_excedencia' => $probabilidadExcedencia,
            'canal' => $canal,
            'eficiencia_sistema' => $eficienciaSistema->eficiencia,
            'eficiencia_conduccion' => $eficienciaConduccion,
            'acciones_extras' => $accionesExtras,
        ]);
    }

    private function calcularDisponibilidadPorSuperficie($oferta, $superficie)
    {
        if ($superficie <= 0) {
            Log::error("Superficie inválida para cálculo de disponibilidad: $superficie");
            return ['disponibilidad' => 0, 'clasificacion' => 'Indefinido', 'valor_clasificacion' => null];
        }

        $disponibilidadPorHa = $oferta / $superficie;
        $clasificacion = $this->clasificarDisponibilidadPorSuperficie($disponibilidadPorHa);

        return [
            'disponibilidad' => $disponibilidadPorHa,
            'clasificacion' => $clasificacion['clasificacion'],
            'valor_clasificacion' => $clasificacion['valor']
        ];
    }

    /**
     * Calcula el número de derechos de acciones de aprovechamiento (DAA) adicionales
     * necesarios para satisfacer la demanda de riego.
     *
     * @param float $demandaTotalTemporada
     * @param float $oferta85
     * @param float $daa
     * @param float $eficienciaConduccion
     * @return float
     */
    private function calcularAccionesExtras($demandaPromedio, $oferta85, $daa, $eficienciaConduccion)
    {
        if ($oferta85 == 0 || $eficienciaConduccion == 0 || $daa == 0) {
            Log::error("Error en cálculo de acciones extras: valores de entrada inválidos.");
            return 0;
        }

        $accionesExtras = ($demandaPromedio - $oferta85) / (($oferta85 / $daa) * $eficienciaConduccion);

        Log::info("Cálculo de acciones extras: Demanda promedio: $demandaPromedio, Oferta 85: $oferta85, DAA: $daa, Eficiencia conducción: $eficienciaConduccion");
        Log::info("Número de acciones adicionales necesarias: $accionesExtras");

        return max(0, $accionesExtras);
    }

    /**
     * Calcula el valor de un percentil específico de un array ordenado.
     *
     * @param array $values
     * @param int $percentile
     * @return float|null
     */
    private function calculatePercentileValue($values, $percentile)
    {
        $total = count($values);
        if ($total === 0) return null;

        $index = round($percentile / 100 * $total) - 1;
        return $values[$index] ?? null;
    }

    private function clasificarDisponibilidadPorSuperficie($disponibilidadPorHa)
    {
        if ($disponibilidadPorHa >= 10000) {
            return ['clasificacion' => 'Muy Bueno', 'valor' => 5];
        } elseif ($disponibilidadPorHa >= 8000) {
            return ['clasificacion' => 'Bueno', 'valor' => 4];
        } elseif ($disponibilidadPorHa >= 6000) {
            return ['clasificacion' => 'Regular', 'valor' => 3];
        } elseif ($disponibilidadPorHa >= 4000) {
            return ['clasificacion' => 'Malo', 'valor' => 2];
        } else {
            return ['clasificacion' => 'Deficiente', 'valor' => 1];
        }
    }

    private function calcularProbabilidadExcedenciaWeibull($seasonalData)
    {
        $clasificaciones = array_column($seasonalData, 'clasificacion_disponibilidad');
        $totalTemporadas = count($clasificaciones);
        $frecuencia = array_count_values($clasificaciones);

        $probabilidadExcedencia = [];
        foreach ($frecuencia as $clasificacion => $cantidad) {
            $probabilidadExcedencia[$clasificacion] = ($cantidad / $totalTemporadas) * 100;
        }

        return $probabilidadExcedencia;
    }

    // Método para clasificar el valor de RWS
    private function clasificarRWS($rws)
    {
        if ($rws >= 1) {
            return 'Adecuado';
        } elseif ($rws >= 0.75) {
            return 'Moderado';
        } elseif ($rws >= 0.5) {
            return 'Crítico';
        } else {
            return 'Severo';
        }
    }

    // Método para calcular el promedio de Kc
    private function calcularPromedioKc($grupo_id, $sectorhidrico_id)
    {
        return Kc::where('grupo_id', $grupo_id)
            ->where('sectorhidrico_id', $sectorhidrico_id)
            ->avg('kc');
    }

    // Método para calcular el promedio de Eto
    private function calcularPromedioEto($sectorhidrico_id)
    {
        return Etomensual::where('sectorhidrico_id', $sectorhidrico_id)
            ->avg('eto');
    }

    // Método para determinar la provincia a partir de la comuna
    private function determinarProvincia($comuna_id)
    {
        $provinciaCode = substr($comuna_id, 0, 2);
        switch ($provinciaCode) {
            case '41': return 'Elqui';
            case '42': return 'Limarí';
            case '43': return 'Choapa';
            default: return null;
        }
    }

    // Método para obtener el periodo de la temporada
    private function getSeasonPeriod($provincia, $periodo)
    {
        $startYear = $this->getYearStart($periodo);
        $endYear = $this->getYearEnd($periodo);
        return [$startYear, $endYear];
    }

    // Método para obtener los meses de la temporada
    private function calculateSeasonMonths($provincia, $startYear)
    {
        $seasonStartMonth = $this->getSeasonStartMonth($provincia);

        $seasonMonths = [];
        for ($month = $seasonStartMonth; $month <= 12; $seasonMonths[] = ['year' => $startYear, 'month' => $month++]);
        for ($month = 1; $month < $seasonStartMonth; $seasonMonths[] = ['year' => $startYear + 1, 'month' => $month++] );

        return $seasonMonths;
    }

    // Método para obtener el mes de inicio de temporada según la provincia
    private function getSeasonStartMonth($provincia)
    {
        return ($provincia === 'Elqui') ? 9 : 5;
    }

    // Método para obtener el año de inicio según el periodo
    private function getYearStart($periodo)
    {
        switch ($periodo) {
            case 'Historico 2000-2024': return 2000;
            case 'Corto plazo 2025-2044': return 2025;
            case 'Mediano plazo 2025-2069': return 2045;
            case 'Largo plazo 2025-2094': return 2094;
            default: return 2000;
        }
    }

    // Método para obtener el año de fin según el periodo
    private function getYearEnd($periodo)
    {
        switch ($periodo) {
            case 'Historico 2000-2024': return 2024;
            case 'Corto plazo 2025-2044': return 2044;
            case 'Mediano plazo 2025-2069': return 2069;
            case 'Largo plazo 2025-2094': return 2094;
            default: return 2024;
        }
    }

    // Método para calcular percentiles
    private function calculatePercentiles($ofertaMensualTotal, $percentiles)
    {
        $total = count($ofertaMensualTotal);
        $percentileValues = [];

        if ($total === 0) {
            foreach ($percentiles as $percentile) {
                $percentileValues["$percentile%"] = null;
            }
            return $percentileValues;
        }

        foreach ($percentiles as $percentile) {
            $index = round($percentile / 100 * $total) - 1;
            if ($index < 0) {
                $index = 0;
            }
            $percentileValues["$percentile%"] = $ofertaMensualTotal[$index] ?? null;
        }

        return $percentileValues;
    }
}
