<?php

namespace App\Http\Controllers;

use App\Models\Canal;
use App\Models\EficienciaRiegoSistema;
use App\Models\Kc;
use App\Models\OfertaSeccRioElquiCsiro;
use App\Models\OfertaSeccRioLimarCsiro;
use App\Models\OfertaSeccRioChoapaCsiro;  
use App\Models\Etomensual;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class CalculoCanalDFController extends Controller
{
    public function calcularDFPorCanal($canal_id, $periodo, $daa, Request $request)
{
    $canal = Canal::with(['sectorhidrico', 'seccionrio', 'comuna', 'subsubcuenca'])->find($canal_id);
    if (!$canal) {
        Log::error("Canal no encontrado: ID {$canal_id}");
        return response()->json(['error' => 'Canal no encontrado'], 404);
    }

    $gruposCultivo = $request->query('grupo_id');
    $superficies = $request->query('superficie');
    $sistemasRiego = explode(',', $request->query('sistema_riego'));

    if (is_string($gruposCultivo)) $gruposCultivo = explode(',', $gruposCultivo);
    if (is_string($superficies)) $superficies = explode(',', $superficies);

    if (!$gruposCultivo || !$superficies || count($gruposCultivo) != count($superficies) || count($sistemasRiego) != count($superficies)) {
        return response()->json(['error' => 'Los grupos de cultivo, superficies y sistemas de riego deben estar emparejados'], 400);
    }

    $eficienciaConduccion = $request->query('eficiencia_conduccion') / 100;
    $provincia = $this->determinarProvincia($canal->comuna_id);
    list($startYear, $endYear) = $this->getSeasonPeriod($provincia, $periodo);

    $seasonalData = [];
    $demandaTotalGeneral = 0;
    $cantidadTemporadas = 0;

    foreach (range($startYear, $endYear - 1) as $year) {
        $ofertaTotalTemporada = 0;
        $demandaTotalTemporada = 0;

        $seasonMonths = $this->calculateSeasonMonths($provincia, $year);

        foreach ($seasonMonths as $period) {
            $currentYear = $period['year'];
            $month = $period['month'];
            $demandaMensualTotal = 0;

            $oferta = OfertaSeccRioElquiCsiro::where('seccionrio_id', $canal->seccionrio_id)
                ->where('fecha', $currentYear)
                ->where('mes', $month)
                ->first() ??
                OfertaSeccRioLimarCsiro::where('seccionrio_id', $canal->seccionrio_id)
                ->where('fecha', $currentYear)
                ->where('mes', $month)
                ->first() ??
                OfertaSeccRioChoapaCsiro::where('seccionrio_id', $canal->seccionrio_id)
                ->where('fecha', $currentYear)
                ->where('mes', $month)
                ->first();

            foreach ($gruposCultivo as $index => $grupo_id) {
                $superficie = $superficies[$index];
                $sistemaRiego = $sistemasRiego[$index];
                $eficienciaSistema = EficienciaRiegoSistema::where('sistema', $sistemaRiego)->first();

                if (!$eficienciaSistema) {
                    Log::error("Eficiencia de riego no encontrada para sistema: {$sistemaRiego}");
                    return response()->json(['error' => 'Eficiencia del sistema de riego no encontrada'], 404);
                }

                $kc = Kc::where('grupo_id', $grupo_id)
                    ->where('sectorhidrico_id', $canal->sectorhidrico_id)
                    ->where('anho', $currentYear)
                    ->where('mes', $month)
                    ->first();

                $eto = Etomensual::where('sectorhidrico_id', $canal->sectorhidrico_id)
                    ->where('anho', $currentYear)
                    ->where('mes', $month)
                    ->first();

                if ($kc && $eto) {
                    $etcBruto = ($kc->kc * $eto->eto * 10 * $superficie) / $eficienciaSistema->eficiencia;
                    $demandaMensualTotal += $etcBruto;
                    
                }
            }

            $demandaTotalTemporada += $demandaMensualTotal;

            if ($oferta) {
                $ofertaMensual = $oferta->ofertahidrica * $daa * $eficienciaConduccion;
                $ofertaTotalTemporada += $ofertaMensual;
                
            }
        }

        if ($demandaTotalTemporada > 0 && $ofertaTotalTemporada > 0) {
            $demandaTotalGeneral += $demandaTotalTemporada;
            $cantidadTemporadas++;


            $seasonalData[] = [
                'temporada' => "{$year}/" . ($year + 1),
                'oferta' => $ofertaTotalTemporada,
                'demanda' => $demandaTotalTemporada,
            ];
        }
    }

    $demandaPromedioTotal_m3 = ($cantidadTemporadas > 0) ? $demandaTotalGeneral / $cantidadTemporadas : 0;
    $demandaPromedioTotal_mm = $demandaPromedioTotal_m3 / 10;
    $ofertas = array_column($seasonalData, 'oferta');
    rsort($ofertas);
    $percentil85_m3 = $this->calcularPercentil($ofertas, 0.85);
    $percentil85_mm = ($percentil85_m3 / (array_sum($superficies) * 10000)) * 1000;


    if ($percentil85_mm > 0) {
        $df = $demandaPromedioTotal_mm / $percentil85_mm;
        $dfClasificado = $this->clasificarDF($df);


        $dfValues = [
            'df' => $df,
            'clasificacion' => $dfClasificado,
        ];
    }

    return response()->json([
        'df_values' => $dfValues ?? [],
    ]);
}






    private function calcularPercentil(array $valores, float $percentil)
    {
        $n = count($valores);
        if ($n === 0) {
            return null;
        }

        $indice = $percentil * ($n - 1);
        $indiceRedondeado = round($indice);
        
        return $valores[$indiceRedondeado];
    }

    private function clasificarDF($df)
    {
        if ($df < 0.6) {
            return 5; // Muy bueno
        } elseif ($df < 0.8) {
            return 4; // Bueno
        } elseif ($df < 1) {
            return 3; // Regular
        } elseif ($df < 1.2) {
            return 2; // Malo
        } else {
            return 1; // Muy malo
        }
    }

    private function determinarProvincia($comuna_id)
    {
        $provinciaCode = substr($comuna_id, 0, 2);
        switch ($provinciaCode) {
            case '41': return 'Elqui';
            case '42': return 'Limarí';
            case '43': return 'Choapa';
            default: return null;
        }
    }

    private function getSeasonPeriod($provincia, $periodo)
    {
        $startYear = $this->getYearStart($periodo);
        $endYear = $this->getYearEnd($periodo);
        return [$startYear, $endYear];
    }

    private function calculateSeasonMonths($provincia, $startYear)
    {
        $seasonStartMonth = $this->getSeasonStartMonth($provincia);
        $seasonMonths = [];
        for ($month = $seasonStartMonth; $month <= 12; $seasonMonths[] = ['year' => $startYear, 'month' => $month++]);
        for ($month = 1; $month < $seasonStartMonth; $seasonMonths[] = ['year' => $startYear + 1, 'month' => $month++]);

        return $seasonMonths;
    }

    private function getSeasonStartMonth($provincia)
    {
        return ($provincia === 'Elqui') ? 9 : 5;
    }

    private function getYearStart($periodo)
    {
        switch ($periodo) {
            case 'Historico 2000-2024': return 2000;
            case 'Corto plazo 2025-2044': return 2025;
            case 'Mediano plazo 2025-2069': return 2045;
            case 'Largo plazo 2025-2094': return 2025;
            default: return 2000;
        }
    }

    private function getYearEnd($periodo)
    {
        switch ($periodo) {
            case 'Historico 2000-2024': return 2024;
            case 'Corto plazo 2025-2044': return 2044;
            case 'Mediano plazo 2025-2069': return 2069;
            case 'Largo plazo 2025-2094': return 2094;
            default: return 2024;
        }
    }
}
