<?php

namespace App\Http\Controllers;

use App\Models\DemandaHidricaM3;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Carbon\Carbon;

class DemandaHidricaM3Controller extends Controller
{
    public function getDemanda(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $anho = $request->input('anho');



        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Demanda Promedio
        $promedio = DB::table($table)
            ->where('nombre', 'like', '%' . $canal . '%')
            ->where('anho', $anho)
            ->avg('demanda_hidrica_bruta_m3');

        // Demanda Máxima
        $maxima = DB::table($table)
            ->where('nombre', $canal)
            ->where('anho', $anho)
            ->max('demanda_hidrica_bruta_m3');

        // Demanda Mínima
        $minima = DB::table($table)
            ->where('nombre', $canal)
            ->where('anho', $anho)
            ->min('demanda_hidrica_bruta_m3');


        return response()->json([
            'promedio' => $promedio,
            'maxima' => $maxima,
            'minima' => $minima,
        ]);
    }

    //Obtener el nombre de los canales
    public function getCanal(Request $request)
    {
        // Obtener los nombres de los canales junto con el nombre de la comuna
        $canales = DB::table('canals')
            ->join('comunas', 'canals.comuna_id', '=', 'comunas.id')
            ->select(DB::raw("CONCAT(canals.nombre, ' - ', comunas.nombre) as nombre"))
            ->pluck('nombre');

        return response()->json($canales);
    }

    //Obtener la provincia a la que pertenece el canal
    public function getProvinciaCanal(Request $request)
    {
        // Obtener el ID del canal desde el request
        $canalId = $request->input('canal');

        // Verificar si el ID del canal fue proporcionado
        if (!$canalId) {
            return response()->json(['error' => 'Debe proporcionar el ID de un canal.'], 400);
        }

        // Obtener el nombre del canal
        $canalName = DB::table('canals')->where('id', $canalId)->value('nombre');

        // Buscar el canal por su ID y obtener la comuna_id
        $canal = DB::table('canals')->where('id', $canalId)->first();

        // Verificar si el canal existe
        if (!$canal) {
            return response()->json(['error' => 'Canal no encontrado.'], 404);
        }

        // Obtener la comuna relacionada al canal
        $comuna = DB::table('comunas')->where('id', $canal->comuna_id)->first();

        // Verificar si la comuna fue encontrada
        if (!$comuna) {
            return response()->json(['error' => 'Comuna no encontrada para el canal.'], 404);
        }

        // Obtener la provincia relacionada a la comuna
        $provincia = DB::table('provincias')->where('id', $comuna->provincia_id)->first();

        // Verificar si la provincia fue encontrada
        if (!$provincia) {
            return response()->json(['error' => 'Provincia no encontrada para la comuna.'], 404);
        }

        // Retornar el id y el nombre de la provincia
        return response()->json([
            'provincia_id' => $provincia->id,
            'provincia_nombre' => $provincia->nombre,
            'comuna_id' => $comuna->id,
            'canal_nombre' => $canalName
        ], 200);
    }



    // Obtener la demanda por año de un canal
    public function getCanalDemanda(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener todas las demandas para el canal seleccionado
        $demandas = DB::table($table)
            ->where('nombre', 'like', '%' . $canal . '%')
            ->get(['anho', 'mes', 'demanda_hidrica_bruta_m3']);

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima->anho,
                'mes' => $nombresMeses[$demandaMaxima->mes], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima->anho,
                'mes' => $nombresMeses[$demandaMinima->mes], // Convertir número de mes a nombre
            ],
        ]);
    }

    // Obtener las demandas de todos los años de un canal
    public function getDemandaByYear(Request $request)
    {
        //en esta función se obtiene la demanda del canal desde el año 2000 hasta el año actual
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);
        $demanda = DB::table($table)
            ->where('nombre', 'like', '%' . $canal . '%')
            ->select('anho', 'demanda_hidrica_bruta_m3')
            ->orderBy('anho', 'asc')
            ->get();
        return response()->json($demanda);
    }

    // obtener la demanda de un unico año
    public function getDemandaByAnho(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $anho = $request->input('anho');

        // Selecciona la tabla por la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);
        $demanda = DB::table($table)
            ->where('nombre', 'like', '%' . $canal . '%')
            ->where('anho', $anho)
            ->get();
        return response()->json($demanda);
    }

    // Obtener la demanda mensual de un año específico
    public function getDemandaMensualPorAnho(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $anho = $request->input('anho');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener la demanda mensual para el año y canal seleccionados
        $demanda = DB::table($table)
            ->where('nombre', 'like', '%' . $canal . '%')
            ->where('anho', $anho)
            ->select('mes', 'demanda_hidrica_bruta_m3')
            ->orderBy('mes', 'asc')
            ->get();
        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });

        return response()->json($demandaFormateada);
    }
    public function getDemandaAnualYMensual(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        $table = DemandaHidricaM3::setTableForProvince($provincia);
        $isElqui = $provincia == '41';

        $result = [];
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $startMonth = $isElqui ? 9 : 5;
            $endMonth = $isElqui ? 8 : 4;
            $endYear = $isElqui ? $year + 1 : $year + 1;

            $query = DB::table($table)
                ->where('nombre', 'like', '%' . $canal . '%')
                ->where(function ($q) use ($year, $endYear, $startMonth, $endMonth) {
                    $q->where(function ($q) use ($year, $startMonth) {
                        $q->where('anho', $year)
                            ->where('mes', '>=', $startMonth);
                    })->orWhere(function ($q) use ($endYear, $endMonth) {
                        $q->where('anho', $endYear)
                            ->where('mes', '<=', $endMonth);
                    });
                });

            $demandaAnual = $query->sum('demanda_hidrica_bruta_m3');
            $demandaAnualFormateada = round($demandaAnual);

            $demandasMensuales = $query
                ->select('anho', 'mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('anho', 'mes')
                ->orderBy('anho', 'asc')
                ->orderBy('mes', 'asc')
                ->get();

            $mensualFormatted = [];
            foreach ($demandasMensuales as $demandaMensual) {
                // Asegurarse de que la propiedad 'demanda_hidrica_bruta_m3' exista
                if (isset($demandaMensual->demanda_hidrica_bruta_m3)) {
                    // Redondear demanda_hidrica_bruta_m3 a entero
                    $demandaMensualFormateada = round($demandaMensual->demanda_hidrica_bruta_m3);
                    $mensualFormatted[] = [
                        'mes' => $demandaMensual->mes,
                        'demanda_hidrica_bruta_m3' => $demandaMensualFormateada
                    ];
                }
            }

            // Formato de temporada: "2000/01" para cada año
            $seasonLabel = "{$year}/" . substr($year + 1, -2);


            $result[] = [
                'anho' => $seasonLabel,
                'demanda_hidrica_bruta_anual' => $demandaAnualFormateada,
                'mensual' => $mensualFormatted
            ];
        }

        return response()->json($result);
    }


    // Obtener la demanda anual y la demanda mensual por cada año
    /*public function getDemandaAnualYMensual(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        $table = DemandaHidricaM3::setTableForProvince($provincia);

        $isElqui = $provincia == '41';

        $result = [];
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $startMonth = $isElqui ? 9 : 5;
            $endMonth = $isElqui ? 8 : 4;
            $endYear = $isElqui ? $year + 1 : $year + 1;

            $query = DB::table($table)
                ->where('nombre', 'like', '%' . $canal . '%')
                ->where(function ($q) use ($year, $endYear, $startMonth, $endMonth) {
                    $q->where(function ($q) use ($year, $startMonth) {
                        $q->where('anho', $year)
                            ->where('mes', '>=', $startMonth);
                    })->orWhere(function ($q) use ($endYear, $endMonth) {
                        $q->where('anho', $endYear)
                            ->where('mes', '<=', $endMonth);
                    });
                });

            $demandaAnual = $query->sum('demanda_hidrica_bruta_m3');

            $demandasMensuales = $query
                ->select('anho', 'mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('anho', 'mes')
                ->orderBy('anho', 'asc')
                ->orderBy('mes', 'asc')
                ->get();

            $mensualFormatted = [];
            foreach ($demandasMensuales as $demandaMensual) {
                $mensualFormatted[] = [
                    'mes' => $demandaMensual->mes,
                    'demanda_hidrica_bruta_m3' => $demandaMensual->demanda_hidrica_bruta_m3
                ];
            }

            $result[] = [
                'anho' => $year,
                'demanda_hidrica_bruta_anual' => $demandaAnual,
                'mensual' => $mensualFormatted
            ];
        }

        return response()->json($result);
    }*/
    /*public function getDemandaAnualYMensual(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener la demanda anual dentro del rango de años seleccionado
        $demandasAnuales = DB::table($table)
            ->where('nombre', 'like', '%' . $canal . '%')
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select('anho', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_anual'))
            ->groupBy('anho')
            ->orderBy('anho', 'asc')
            ->get();

        $result = [];
        foreach ($demandasAnuales as $demandaAnual) {
            // Obtener la demanda mensual solo para los años dentro del rango
            $demandasMensuales = DB::table($table)
                ->where('nombre', 'like', '%' . $canal . '%')
                ->where('anho', $demandaAnual->anho)
                ->select('mes', 'demanda_hidrica_bruta_m3')
                ->orderBy('mes', 'asc')
                ->get();

            // Crear un array con el año y la demanda mensual
            $result[] = [
                'anho' => $demandaAnual->anho,
                'demanda_hidrica_bruta_anual' => $demandaAnual->demanda_hidrica_bruta_anual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }*/

    public function getProvinciaDemanda(Request $request)
    {
        $provincia = $request->input('provincia');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener todas las demandas a nivel provincial
        $demandas = DB::table($table)
            ->select('anho', 'mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('anho', 'mes')
            ->get();

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima->anho,
                'mes' => $nombresMeses[$demandaMaxima->mes], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima->anho,
                'mes' => $nombresMeses[$demandaMinima->mes], // Convertir número de mes a nombre
            ],
        ]);
    }

    public function getDemandaMensualPorAnhoProvincia(Request $request)
    {
        $provincia = $request->input('provincia');
        $anho = $request->input('anho');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener la demanda mensual para el año seleccionado a nivel provincial
        $demanda = DB::table($table)
            ->where('anho', $anho)
            ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('mes')
            ->orderBy('mes', 'asc')
            ->get();
        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });

        return response()->json($demandaFormateada);
    }

    public function getDemandaAnualYMensualProvincia(Request $request)
    {
        $provincia = $request->input('provincia');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        $table = DemandaHidricaM3::setTableForProvince($provincia);
        $isElqui = $provincia == '41';

        $result = [];
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $startMonth = $isElqui ? 9 : 5;
            $endMonth = $isElqui ? 8 : 4;
            $endYear = $year + 1;

            $query = DB::table($table)
                ->where(function ($q) use ($year, $endYear, $startMonth, $endMonth) {
                    $q->where(function ($q) use ($year, $startMonth) {
                        $q->where('anho', $year)
                            ->where('mes', '>=', $startMonth);
                    })->orWhere(function ($q) use ($endYear, $endMonth) {
                        $q->where('anho', $endYear)
                            ->where('mes', '<=', $endMonth);
                    });
                });

            $demandaAnual = $query->sum('demanda_hidrica_bruta_m3');
            $demandaAnualFormateada = round($demandaAnual);

            $demandasMensuales = $query
                ->select('anho', 'mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('anho', 'mes')
                ->orderBy('anho', 'asc')
                ->orderBy('mes', 'asc')
                ->get();

            $mensualFormatted = [];
            /*foreach ($demandasMensuales as $demandaMensual) {
                $mensualFormatted[] = [
                    'mes' => $demandaMensual->mes,
                    'demanda_hidrica_bruta_m3' => $demandaMensual->demanda_hidrica_bruta_m3
                ];
            }*/
            foreach ($demandasMensuales as $demandaMensual) {
                // Asegurarse de que la propiedad 'demanda_hidrica_bruta_m3' exista
                if (isset($demandaMensual->demanda_hidrica_bruta_m3)) {
                    // Redondear demanda_hidrica_bruta_m3 a entero
                    $demandaMensualFormateada = round($demandaMensual->demanda_hidrica_bruta_m3);
                    $mensualFormatted[] = [
                        'mes' => $demandaMensual->mes,
                        'demanda_hidrica_bruta_m3' => $demandaMensualFormateada
                    ];
                }
            }

            $seasonLabel = "{$year}/" . substr($year + 1, -2);

            $result[] = [
                'anho' => $seasonLabel,
                'demanda_hidrica_bruta_anual' => $demandaAnualFormateada,
                'mensual' => $mensualFormatted
            ];
        }

        return response()->json($result);
    }
    /*public function getDemandaAnualYMensualProvincia(Request $request)
    {
        $provincia = $request->input('provincia');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener la demanda anual dentro del rango de años seleccionado
        $demandasAnuales = DB::table($table)
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select('anho', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_anual'))
            ->groupBy('anho')
            ->orderBy('anho', 'asc')
            ->get();

        $result = [];
        foreach ($demandasAnuales as $demandaAnual) {
            // Obtener la demanda mensual solo para los años dentro del rango
            $demandasMensuales = DB::table($table)
                ->where('anho', $demandaAnual->anho)
                ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('mes')
                ->orderBy('mes', 'asc')
                ->get();

            // Crear un array con el año y la demanda mensual
            $result[] = [
                'anho' => $demandaAnual->anho,
                'demanda_hidrica_bruta_anual' => $demandaAnual->demanda_hidrica_bruta_anual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }*/


    public function getComunaDemanda(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener los nombres de los canales asociados a la comuna
        $canales = DB::table('canals')
            ->where('comuna_id', $comunaId)
            ->pluck('nombre');

        // Obtener todas las demandas para los canales de la comuna
        $demandas = DB::table($table)
            ->whereIn('nombre', $canales)  // Filtrar por los nombres de los canales
            ->select('anho', 'mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('anho', 'mes')
            ->get();

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima->anho,
                'mes' => $nombresMeses[$demandaMaxima->mes], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima->anho,
                'mes' => $nombresMeses[$demandaMinima->mes], // Convertir número de mes a nombre
            ],
        ]);
    }

    public function getDemandaMensualPorAnhoComuna(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');
        $temporadaInicio = $request->input('anho'); // Año de inicio de la temporada

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener los nombres de los canales con caché
        $canales = cache()->remember("canales_comuna_{$comunaId}", 3600, function () use ($comunaId) {
            return DB::table('canals')
                ->where('comuna_id', $comunaId)
                ->pluck('nombre')
                ->toArray();
        });

        // Si no hay canales, retornar array vacío
        if (empty($canales)) {
            return response()->json([]);
        }

        // Definir el mes de inicio y fin según la provincia
        $isElqui = $provincia == '41';
        $startMonth = $isElqui ? 9 : 5;  // Septiembre para Elqui, Mayo para Choapa y Limarí
        $endMonth = $isElqui ? 8 : 4;    // Agosto para Elqui, Abril para Choapa y Limarí
        $temporadaFin = $temporadaInicio + 1;

        // Obtener la demanda mensual para la temporada seleccionada
        $demanda = DB::table($table)
            ->whereIn('nombre', $canales)
            ->where(function ($query) use ($temporadaInicio, $temporadaFin, $startMonth, $endMonth) {
                $query->where(function ($q) use ($temporadaInicio, $startMonth) {
                    $q->where('anho', $temporadaInicio)
                        ->where('mes', '>=', $startMonth);
                })->orWhere(function ($q) use ($temporadaFin, $endMonth) {
                    $q->where('anho', $temporadaFin)
                        ->where('mes', '<=', $endMonth);
                });
            })
            ->select('anho', 'mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('anho', 'mes')
            ->orderBy('anho')
            ->orderBy('mes')
            ->get();

        // Formatear y ordenar los resultados
        $demandaFormateada = $demanda->map(function ($item) {
            return [
                'mes' => $item->mes,
                'demanda_hidrica_bruta_m3' => round($item->demanda_hidrica_bruta_m3)
            ];
        });

        // Ordenar los meses según la temporada
        $demandaOrdenada = $demandaFormateada->sort(function ($a, $b) use ($startMonth) {
            $monthA = $a['mes'];
            $monthB = $b['mes'];

            // Ajustar los meses para que el mes de inicio sea el primero
            $monthA = $monthA >= $startMonth ? $monthA - $startMonth : $monthA + (12 - $startMonth);
            $monthB = $monthB >= $startMonth ? $monthB - $startMonth : $monthB + (12 - $startMonth);

            return $monthA - $monthB;
        })->values();

        return response()->json($demandaOrdenada);
    }

    /*public function getDemandaMensualPorAnhoComuna(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');
        $anho = $request->input('anho');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener los nombres de los canales asociados a la comuna
        $canales = DB::table('canals')
            ->where('comuna_id', $comunaId)
            ->pluck('nombre');

        // Obtener la demanda mensual para el año seleccionado a nivel de comuna
        $demanda = DB::table($table)
            ->whereIn('nombre', $canales)
            ->where('anho', $anho)
            ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('mes')
            ->orderBy('mes', 'asc')
            ->get();
        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });

        return response()->json($demandaFormateada);
    }*/

    public function getDemandaAnualYMensualComuna(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener los nombres de los canales en una sola consulta y cachear el resultado
        $canales = cache()->remember("canales_comuna_{$comunaId}", 3600, function () use ($comunaId) {
            return DB::table('canals')
                ->where('comuna_id', $comunaId)
                ->pluck('nombre')
                ->toArray();
        });

        // Si no hay canales, retornar array vacío
        if (empty($canales)) {
            return response()->json([]);
        }

        // Definir el mes de inicio y fin según la provincia
        $isElqui = $provincia == '41';
        $startMonth = $isElqui ? 9 : 5;  // Septiembre para Elqui, Mayo para Choapa y Limarí
        $endMonth = $isElqui ? 8 : 4;    // Agosto para Elqui, Abril para Choapa y Limarí

        $result = [];
        $yearsToQuery = [];

        // Preparar los rangos de años necesarios para cubrir todas las temporadas
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $yearsToQuery[] = $year;
            $yearsToQuery[] = $year + 1;
        }
        $yearsToQuery = array_unique($yearsToQuery);

        // Realizar una única consulta para obtener todos los datos
        $allData = DB::table($table)
            ->select(
                'anho',
                'mes',
                DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_total')
            )
            ->whereIn('nombre', $canales)
            ->whereIn('anho', $yearsToQuery)
            ->groupBy('anho', 'mes')
            ->orderBy('anho')
            ->orderBy('mes')
            ->get();

        // Procesar los datos por temporada
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $endYear = $year + 1;
            $seasonLabel = "{$year}/" . substr($endYear, -2);

            // Filtrar datos para la temporada actual
            $seasonData = $allData->filter(function ($row) use ($year, $endYear, $startMonth, $endMonth) {
                return ($row->anho == $year && $row->mes >= $startMonth) ||
                    ($row->anho == $endYear && $row->mes <= $endMonth);
            });

            if ($seasonData->isEmpty()) {
                continue;
            }

            // Calcular demanda anual para la temporada
            $demandaAnual = $seasonData->sum('demanda_total');

            // Preparar datos mensuales
            $mensualFormatted = $seasonData->map(function ($row) {
                return [
                    'mes' => $row->mes,
                    'demanda_hidrica_bruta_m3' => round($row->demanda_total)
                ];
            })->values()->toArray();

            // Ordenar los meses según la temporada
            usort($mensualFormatted, function ($a, $b) use ($startMonth) {
                $monthA = $a['mes'];
                $monthB = $b['mes'];

                // Ajustar los meses para que el mes de inicio sea el primero
                $monthA = $monthA >= $startMonth ? $monthA - $startMonth : $monthA + (12 - $startMonth);
                $monthB = $monthB >= $startMonth ? $monthB - $startMonth : $monthB + (12 - $startMonth);

                return $monthA - $monthB;
            });

            $result[] = [
                'anho' => $seasonLabel,
                'demanda_hidrica_bruta_anual' => round($demandaAnual),
                'mensual' => $mensualFormatted
            ];
        }

        return response()->json($result);
    }

    /*public function getDemandaAnualYMensualComuna(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la provincia
        $table = DemandaHidricaM3::setTableForProvince($provincia);

        // Obtener los nombres de los canales asociados a la comuna
        $canales = DB::table('canals')
            ->where('comuna_id', $comunaId)
            ->pluck('nombre');

        // Obtener la demanda anual dentro del rango de años seleccionado
        $demandasAnuales = DB::table($table)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select('anho', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_anual'))
            ->groupBy('anho')
            ->orderBy('anho', 'asc')
            ->get();

        $result = [];
        foreach ($demandasAnuales as $demandaAnual) {
            // Obtener la demanda mensual solo para los años dentro del rango
            $demandasMensuales = DB::table($table)
                ->whereIn('nombre', $canales)
                ->where('anho', $demandaAnual->anho)
                ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('mes')
                ->orderBy('mes', 'asc')
                ->get();

            // Crear un array con el año y la demanda mensual
            $result[] = [
                'anho' => $demandaAnual->anho,
                'demanda_hidrica_bruta_anual' => $demandaAnual->demanda_hidrica_bruta_anual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }*/


    public function getCuencaDemanda(Request $request)
    {
        $cuencaId = $request->input('cuenca');

        // Obtener los canales asociados a la cuenca (a través de subcuencas y subsubcuencas)
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->join('subcuencas', 'subsubcuencas.subcuenca_id', '=', 'subcuencas.id')
            ->where('subcuencas.cuenca_id', $cuencaId)
            ->pluck('canals.nombre');

        // Obtener todas las demandas de las tres tablas para los canales de la cuenca
        $demandas = DemandaHidricaM3::getCombinedDemandaHidrica()
            ->whereIn('nombre', $canales)  // Filtrar por los nombres de los canales
            ->groupBy(['anho', 'mes'])
            ->map(function ($group) {
                return [
                    'anho' => $group->first()->anho,
                    'mes' => $group->first()->mes,
                    'demanda_hidrica_bruta_m3' => $group->sum('demanda_hidrica_bruta_m3'),
                ];
            });

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima['anho'],
                'mes' => $nombresMeses[$demandaMaxima['mes']], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima['anho'],
                'mes' => $nombresMeses[$demandaMinima['mes']], // Convertir número de mes a nombre
            ],
        ]);
    }

    public function getSubcuencaDemanda(Request $request)
    {
        $subcuencaId = $request->input('subcuenca');

        // Obtener los canales asociados a la subcuenca (a través de subsubcuencas)
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->where('subsubcuencas.subcuenca_id', $subcuencaId)
            ->pluck('canals.nombre');

        // Obtener todas las demandas de las tres tablas para los canales de la subcuenca
        $demandas = DemandaHidricaM3::getCombinedDemandaHidrica()
            ->whereIn('nombre', $canales)  // Filtrar por los nombres de los canales
            ->groupBy(['anho', 'mes'])
            ->map(function ($group) {
                return [
                    'anho' => $group->first()->anho,
                    'mes' => $group->first()->mes,
                    'demanda_hidrica_bruta_m3' => $group->sum('demanda_hidrica_bruta_m3'),
                ];
            });

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima['anho'],
                'mes' => $nombresMeses[$demandaMaxima['mes']], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima['anho'],
                'mes' => $nombresMeses[$demandaMinima['mes']], // Convertir número de mes a nombre
            ],
        ]);
    }

    public function getDemandaMensualPorAnhoCuenca(Request $request)
    {
        $cuencaId = $request->input('cuenca');
        $anho = $request->input('anho');

        // Selecciona la tabla según la cuenca
        $table = DemandaHidricaM3::getCombinedDemandaHidricaMonthYear($cuencaId);

        // Obtener los nombres de los canales asociados a la cuenca
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->join('subcuencas', 'subsubcuencas.subcuenca_id', '=', 'subcuencas.id')
            ->where('subcuencas.cuenca_id', $cuencaId)
            ->pluck('canals.nombre');

        // Obtener la demanda mensual para el año seleccionado a nivel de cuenca
        $demanda = DB::table($table)
            ->whereIn('nombre', $canales)
            ->where('anho', $anho)
            ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('mes')
            ->orderBy('mes', 'asc')
            ->get();

        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });


        return response()->json($demandaFormateada);
    }


    public function getDemandaMensualPorAnhoSubcuenca(Request $request)
    {
        $subcuencaId = $request->input('subcuenca');
        $anho = $request->input('anho');

        // Selecciona la tabla según la cuenca relacionada a la subcuenca
        $cuencaId = DB::table('subcuencas')->where('id', $subcuencaId)->value('cuenca_id');
        $table = DemandaHidricaM3::getCombinedDemandaHidricaMonthYear($cuencaId);

        // Obtener los nombres de los canales asociados a la subcuenca
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->where('subsubcuencas.subcuenca_id', $subcuencaId)
            ->pluck('canals.nombre');

        // Obtener la demanda mensual para el año seleccionado a nivel de subcuenca
        $demanda = DB::table($table)
            ->whereIn('nombre', $canales)
            ->where('anho', $anho)
            ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('mes')
            ->orderBy('mes', 'asc')
            ->get();
        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });

        return response()->json($demandaFormateada);
    }

    public function getDemandaAnualYMensualCuenca(Request $request)
    {
        $cuencaId = $request->input('cuenca');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la cuenca
        $table = DemandaHidricaM3::getCombinedDemandaHidricaMonthYear($cuencaId);

        // Determinar si es Elqui basado en la cuenca
        $isElqui = in_array($cuencaId, ['40', '41', '42', '43']);

        // Obtener los nombres de los canales asociados a la cuenca
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->join('subcuencas', 'subsubcuencas.subcuenca_id', '=', 'subcuencas.id')
            ->where('subcuencas.cuenca_id', $cuencaId)
            ->pluck('canals.nombre');

        // Ajustar el rango de años para incluir el año siguiente al final del rango
        $adjustedRangeEnd = $rangeEnd + 1;

        // Obtener demanda anual y mensual en una sola consulta
        $demandas = DB::table($table)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $adjustedRangeEnd])
            ->select(
                'anho',
                'mes',
                DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3')
            )
            ->groupBy('anho', 'mes')
            ->orderBy('anho', 'asc')
            ->orderBy('mes', 'asc')
            ->get();

        // Organizar el resultado en la estructura deseada
        $result = [];
        foreach ($demandas as $demanda) {
            $year = $demanda->anho;
            $month = $demanda->mes;

            // Determinar a qué temporada pertenece este mes
            if ($isElqui) {
                $seasonYear = ($month >= 9) ? $year : $year - 1;
            } else {
                $seasonYear = ($month >= 5) ? $year : $year - 1;
            }

            // Formato de temporada: "2000/01" para cada temporada
            $seasonLabel = "{$seasonYear}/" . substr($seasonYear + 1, -2);

            if (!isset($result[$seasonYear])) {
                $result[$seasonYear] = [
                    'anho' => $seasonLabel,
                    'demanda_hidrica_bruta_anual' => 0,
                    'mensual' => []
                ];
            }

            // Sumar la demanda anual
            //$result[$seasonYear]['demanda_hidrica_bruta_anual'] += $demanda->demanda_hidrica_bruta_m3;
            $result[$seasonYear]['demanda_hidrica_bruta_anual'] += round($demanda->demanda_hidrica_bruta_m3);
            // Agregar la demanda mensual
            $result[$seasonYear]['mensual'][] = [
                'mes' => $month,
                //'demanda_hidrica_bruta_m3' => $demanda->demanda_hidrica_bruta_m3
                'demanda_hidrica_bruta_m3' => round($demanda->demanda_hidrica_bruta_m3)
            ];
        }

        // Filtrar los resultados para incluir solo las temporadas completas dentro del rango solicitado
        $filteredResult = array_filter($result, function ($item) use ($rangeStart, $rangeEnd) {
            return $item['anho'] >= $rangeStart && $item['anho'] <= $rangeEnd;
        });

        return response()->json(array_values($filteredResult));
    }



    /*public function getDemandaAnualYMensualCuenca(Request $request)
    {
        $cuencaId = $request->input('cuenca');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la cuenca
        $table = DemandaHidricaM3::getCombinedDemandaHidricaMonthYear($cuencaId);

        // Obtener los nombres de los canales asociados a la cuenca
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->join('subcuencas', 'subsubcuencas.subcuenca_id', '=', 'subcuencas.id')
            ->where('subcuencas.cuenca_id', $cuencaId)
            ->pluck('canals.nombre');

        // Obtener demanda anual y mensual en una sola consulta
        $demandas = DB::table($table)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select(
                'anho',
                'mes',
                DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3')
            )
            ->groupBy('anho', 'mes')
            ->orderBy('anho', 'asc')
            ->orderBy('mes', 'asc')
            ->get();

        // Organizar el resultado en la estructura deseada
        $result = [];
        foreach ($demandas as $demanda) {
            if (!isset($result[$demanda->anho])) {
                $result[$demanda->anho] = [
                    'anho' => $demanda->anho,
                    'demanda_hidrica_bruta_anual' => 0,
                    'mensual' => []
                ];
            }

            // Sumar la demanda anual
            $result[$demanda->anho]['demanda_hidrica_bruta_anual'] += $demanda->demanda_hidrica_bruta_m3;

            // Agregar la demanda mensual
            $result[$demanda->anho]['mensual'][] = [
                'mes' => $demanda->mes,
                'demanda_hidrica_bruta_m3' => $demanda->demanda_hidrica_bruta_m3
            ];
        }

        return response()->json(array_values($result));
    }*/

    public function getDemandaAnualYMensualSubcuenca(Request $request)
    {
        $subcuencaId = $request->input('subcuenca');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Obtener el cuenca_id relacionado a la subcuenca
        $cuencaId = DB::table('subcuencas')->where('id', $subcuencaId)->value('cuenca_id');

        // Selecciona la tabla según la cuenca relacionada a la subcuenca
        $table = DemandaHidricaM3::getCombinedDemandaHidricaMonthYear($cuencaId);

        // Determinar si es Elqui basado en la cuenca
        $isElqui = in_array($cuencaId, ['40', '41', '42', '43']);

        // Obtener los nombres de los canales asociados a la subcuenca
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->where('subsubcuencas.subcuenca_id', $subcuencaId)
            ->pluck('canals.nombre');

        // Ajustar el rango de años para incluir el año siguiente al final del rango
        $adjustedRangeEnd = $rangeEnd + 1;

        // Obtener demanda anual y mensual en una sola consulta
        $demandas = DB::table($table)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $adjustedRangeEnd])
            ->select(
                'anho',
                'mes',
                DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3')
            )
            ->groupBy('anho', 'mes')
            ->orderBy('anho', 'asc')
            ->orderBy('mes', 'asc')
            ->get();

        // Organizar el resultado en la estructura deseada
        $result = [];
        foreach ($demandas as $demanda) {
            $year = $demanda->anho;
            $month = $demanda->mes;

            // Determinar a qué temporada pertenece este mes
            if ($isElqui) {
                $seasonYear = ($month >= 9) ? $year : $year - 1;
            } else {
                $seasonYear = ($month >= 5) ? $year : $year - 1;
            }
            // Formato de temporada: "2000/01" para cada temporada
            $seasonLabel = "{$seasonYear}/" . substr($seasonYear + 1, -2);

            if (!isset($result[$seasonYear])) {
                $result[$seasonYear] = [
                    'anho' => $seasonLabel,
                    'demanda_hidrica_bruta_anual' => 0,
                    'mensual' => []
                ];
            }

            // Sumar la demanda anual
            //$result[$seasonYear]['demanda_hidrica_bruta_anual'] += $demanda->demanda_hidrica_bruta_m3;
            $result[$seasonYear]['demanda_hidrica_bruta_anual'] += round($demanda->demanda_hidrica_bruta_m3);
            // Agregar la demanda mensual
            $result[$seasonYear]['mensual'][] = [
                'mes' => $month,
                //'demanda_hidrica_bruta_m3' => $demanda->demanda_hidrica_bruta_m3
                'demanda_hidrica_bruta_m3' => round($demanda->demanda_hidrica_bruta_m3)
            ];
        }

        // Filtrar los resultados para incluir solo las temporadas completas dentro del rango solicitado
        $filteredResult = array_filter($result, function ($item) use ($rangeStart, $rangeEnd) {
            return $item['anho'] >= $rangeStart && $item['anho'] <= $rangeEnd;
        });

        return response()->json(array_values($filteredResult));
    }

    /*public function getDemandaAnualYMensualSubcuenca(Request $request)
    {
        $subcuencaId = $request->input('subcuenca');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Obtener el cuenca_id relacionado a la subcuenca
        $cuencaId = DB::table('subcuencas')->where('id', $subcuencaId)->value('cuenca_id');

        // Selecciona la tabla según la cuenca relacionada a la subcuenca
        $table = DemandaHidricaM3::getCombinedDemandaHidricaMonthYear($cuencaId);

        // Obtener los nombres de los canales asociados a la subcuenca
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->where('subsubcuencas.subcuenca_id', $subcuencaId)
            ->pluck('canals.nombre');

        // Obtener la demanda anual dentro del rango de años seleccionado
        $demandasAnuales = DB::table($table)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select('anho', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_anual'))
            ->groupBy('anho')
            ->orderBy('anho', 'asc')
            ->get();

        $result = [];
        foreach ($demandasAnuales as $demandaAnual) {
            // Obtener la demanda mensual para cada año
            $demandasMensuales = DB::table($table)
                ->whereIn('nombre', $canales)
                ->where('anho', $demandaAnual->anho)
                ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('mes')
                ->orderBy('mes', 'asc')
                ->get();

            $result[] = [
                'anho' => $demandaAnual->anho,
                'demanda_hidrica_bruta_anual' => $demandaAnual->demanda_hidrica_bruta_anual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }*/

    public function getDemandaMensualPorAnhoOrganizacion(Request $request)
    {
        $organizacionId = $request->input('organizacion');
        $anho = $request->input('anho');

        // Obtener los nombres de los canales asociados a la organización (a través de ríos y secciones de ríos)
        $canales = DB::table('canals')
            ->join('seccionrios', 'canals.seccionrio_id', '=', 'seccionrios.id')
            ->join('rios', 'seccionrios.rio_id', '=', 'rios.id')
            ->where('rios.organizacion_id', $organizacionId)
            ->pluck('canals.nombre');

        // Obtener la demanda mensual para el año seleccionado a nivel de organización
        $demanda = DB::table(DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($organizacionId))
            ->whereIn('nombre', $canales)
            ->where('anho', $anho)
            ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('mes')
            ->orderBy('mes', 'asc')
            ->get();

        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });

        return response()->json($demandaFormateada);
    }

    public function getDemandaMensualPorAnhoRio(Request $request)
    {
        $seccionRioId = $request->input('rio'); // Obtener seccionrio_id del request
        $anho = $request->input('anho'); // Obtener el año del request

        // Obtener los nombres de los canales asociados a la sección del río
        $canales = DB::table('canals')
            ->where('canals.seccionrio_id', $seccionRioId) // Filtrar por seccionrio_id
            ->pluck('canals.nombre');

        // Obtener la demanda mensual para el año seleccionado a nivel de sección del río
        $demanda = DB::table(DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($seccionRioId))
            ->whereIn('nombre', $canales)
            ->where('anho', $anho)
            ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
            ->groupBy('mes')
            ->orderBy('mes', 'asc')
            ->get();

        // Formatear cada demanda_hidrica_bruta_m3 a entero
        $demandaFormateada = $demanda->map(function ($item) {
            // Redondear el valor de demanda_hidrica_bruta_m3
            $item->demanda_hidrica_bruta_m3 = round($item->demanda_hidrica_bruta_m3);
            return $item;
        });

        return response()->json($demandaFormateada);
    }

    public function getDemandaAnualYMensualOrganizacion(Request $request)
    {
        $organizacionId = $request->input('organizacion');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Obtener los nombres de los canales asociados a la organización
        $canales = DB::table('canals')
            ->join('seccionrios', 'canals.seccionrio_id', '=', 'seccionrios.id')
            ->join('rios', 'seccionrios.rio_id', '=', 'rios.id')
            ->where('rios.organizacion_id', $organizacionId)
            ->pluck('canals.nombre');

        $tableName = DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($organizacionId);

        // Determinar la temporada basada en la organización
        $isElqui = in_array($organizacionId, ['101', '102']);
        $startMonth = $isElqui ? 9 : 5; // Septiembre para Elqui, Mayo para los demás

        // Ajustar el rango de años para cubrir las temporadas completas
        $adjustedRangeStart = $isElqui ? $rangeStart : $rangeStart - 1;
        $adjustedRangeEnd = $isElqui ? $rangeEnd + 1 : $rangeEnd;

        // Obtener todos los datos de una sola vez
        $allData = DB::table($tableName)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$adjustedRangeStart, $adjustedRangeEnd])
            ->select('anho', 'mes', 'demanda_hidrica_bruta_m3')
            ->get();

        $result = [];
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $seasonStart = $year;
            $seasonEnd = $year + 1;

            // Filtrar datos para la temporada actual
            $seasonData = $allData->filter(function ($item) use ($seasonStart, $seasonEnd, $startMonth) {
                if ($item->anho == $seasonStart && $item->mes >= $startMonth) return true;
                if ($item->anho == $seasonEnd && $item->mes < $startMonth) return true;
                return false;
            });

            // Calcular demanda anual
            //$demandaAnual = $seasonData->sum('demanda_hidrica_bruta_m3');
            $demandaAnual = round($seasonData->sum('demanda_hidrica_bruta_m3'));
            // Preparar demanda mensual
            $demandasMensuales = [];
            for ($i = 1; $i <= 12; $i++) {
                $monthData = $seasonData->filter(function ($item) use ($i, $startMonth) {
                    $adjustedMonth = ($item->mes - $startMonth + 12) % 12 + 1;
                    return $adjustedMonth == $i;
                });
                $demandasMensuales[] = [
                    'mes' => $i,
                    //'demanda_hidrica_bruta_m3' => $monthData->sum('demanda_hidrica_bruta_m3')
                    'demanda_hidrica_bruta_m3' => round($monthData->sum('demanda_hidrica_bruta_m3'))
                ];
            }

            // Agregar al resultado
            $result[] = [
                'anho' => "{$year}/" . substr($seasonEnd, -2),
                'demanda_hidrica_bruta_anual' => $demandaAnual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }

    /*public function getDemandaAnualYMensualOrganizacion(Request $request)
    {
        $organizacionId = $request->input('organizacion');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Obtener los nombres de los canales asociados a la organización
        $canales = DB::table('canals')
            ->join('seccionrios', 'canals.seccionrio_id', '=', 'seccionrios.id')
            ->join('rios', 'seccionrios.rio_id', '=', 'rios.id')
            ->where('rios.organizacion_id', $organizacionId)
            ->pluck('canals.nombre');

        // Obtener la demanda anual dentro del rango de años seleccionado
        $demandasAnuales = DB::table(DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($organizacionId))
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select('anho', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_anual'))
            ->groupBy('anho')
            ->orderBy('anho', 'asc')
            ->get();

        $result = [];
        foreach ($demandasAnuales as $demandaAnual) {
            // Obtener la demanda mensual solo para los años dentro del rango
            $demandasMensuales = DB::table(DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($organizacionId))
                ->whereIn('nombre', $canales)
                ->where('anho', $demandaAnual->anho)
                ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('mes')
                ->orderBy('mes', 'asc')
                ->get();

            // Crear un array con el año y la demanda mensual
            $result[] = [
                'anho' => $demandaAnual->anho,
                'demanda_hidrica_bruta_anual' => $demandaAnual->demanda_hidrica_bruta_anual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }*/

    public function getDemandaAnualYMensualRio(Request $request)
    {
        $seccionRioId = $request->input('rio');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Obtener el ID de la organización para determinar la provincia
        $organizacionId = DB::table('rios')
            ->join('seccionrios', 'rios.id', '=', 'seccionrios.rio_id')
            ->where('seccionrios.id', $seccionRioId)
            ->value('rios.organizacion_id');

        // Determinar el inicio de temporada según la provincia
        $isElqui = in_array($organizacionId, ['101', '102']);
        $startMonth = $isElqui ? 9 : 5; // Septiembre para Elqui, Mayo para Limarí y Choapa

        // Obtener los nombres de los canales asociados a la sección del río
        $canales = DB::table('canals')
            ->where('canals.seccionrio_id', $seccionRioId)
            ->pluck('canals.nombre');

        // Ajustar el rango de años para cubrir temporadas completas
        $adjustedRangeStart = $isElqui ? $rangeStart : $rangeStart - 1;
        $adjustedRangeEnd = $isElqui ? $rangeEnd + 1 : $rangeEnd;

        // Obtener todos los datos de una sola vez
        $tableName = DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($organizacionId);
        $allData = DB::table($tableName)
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$adjustedRangeStart, $adjustedRangeEnd])
            ->select('anho', 'mes', 'demanda_hidrica_bruta_m3')
            ->get();

        $result = [];
        for ($year = $rangeStart; $year <= $rangeEnd; $year++) {
            $seasonStart = $year;
            $seasonEnd = $year + 1;

            // Filtrar datos para la temporada actual
            $seasonData = $allData->filter(function ($item) use ($seasonStart, $seasonEnd, $startMonth) {
                if ($item->anho == $seasonStart && $item->mes >= $startMonth) return true;
                if ($item->anho == $seasonEnd && $item->mes < $startMonth) return true;
                return false;
            });

            // Calcular la demanda anual para la temporada
            //$demandaAnual = $seasonData->sum('demanda_hidrica_bruta_m3');
            $demandaAnual = round($seasonData->sum('demanda_hidrica_bruta_m3'));


            // Preparar demanda mensual
            $demandasMensuales = [];
            for ($i = 1; $i <= 12; $i++) {
                $monthData = $seasonData->filter(function ($item) use ($i, $startMonth) {
                    $adjustedMonth = ($item->mes - $startMonth + 12) % 12 + 1;
                    return $adjustedMonth == $i;
                });
                $demandasMensuales[] = [
                    'mes' => $i,
                    //'demanda_hidrica_bruta_m3' => $monthData->sum('demanda_hidrica_bruta_m3')
                    'demanda_hidrica_bruta_m3' => round($monthData->sum('demanda_hidrica_bruta_m3'))
                ];
            }

            // Agregar al resultado
            $result[] = [
                'anho' => "{$year}/" . substr($seasonEnd, -2),
                'demanda_hidrica_bruta_anual' => $demandaAnual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }

    /*public function getDemandaAnualYMensualRio(Request $request)
    {
        $seccionRioId = $request->input('rio'); // Obtener seccionrio_id del request
        $rangeStart = $request->input('rangeStart'); // Rango de inicio
        $rangeEnd = $request->input('rangeEnd'); // Rango de fin

        // Obtener los nombres de los canales asociados a la sección del río
        $canales = DB::table('canals')
            ->where('canals.seccionrio_id', $seccionRioId) // Filtrar por seccionrio_id
            ->pluck('canals.nombre');

        // Obtener la demanda anual dentro del rango de años seleccionado
        $demandasAnuales = DB::table(DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($seccionRioId))
            ->whereIn('nombre', $canales)
            ->whereBetween('anho', [$rangeStart, $rangeEnd])
            ->select('anho', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_anual'))
            ->groupBy('anho')
            ->orderBy('anho', 'asc')
            ->get();

        $result = [];
        foreach ($demandasAnuales as $demandaAnual) {
            // Obtener la demanda mensual solo para los años dentro del rango
            $demandasMensuales = DB::table(DemandaHidricaM3::getCombinedDemandaHidricaMonthYearOrganization($seccionRioId))
                ->whereIn('nombre', $canales)
                ->where('anho', $demandaAnual->anho)
                ->select('mes', DB::raw('SUM(demanda_hidrica_bruta_m3) as demanda_hidrica_bruta_m3'))
                ->groupBy('mes')
                ->orderBy('mes', 'asc')
                ->get();

            // Crear un array con el año y la demanda mensual
            $result[] = [
                'anho' => $demandaAnual->anho,
                'demanda_hidrica_bruta_anual' => $demandaAnual->demanda_hidrica_bruta_anual,
                'mensual' => $demandasMensuales
            ];
        }

        return response()->json($result);
    }*/


    public function getOrganizacionDemanda(Request $request)
    {
        $organizacionId = $request->input('organizacion');

        // Obtener los canales asociados a la organización
        $canales = DB::table('canals')
            ->join('seccionrios', 'canals.seccionrio_id', '=', 'seccionrios.id')
            ->join('rios', 'seccionrios.rio_id', '=', 'rios.id')
            ->where('rios.organizacion_id', $organizacionId)
            ->pluck('canals.nombre');

        // Obtener todas las demandas de las tres tablas para los canales de la organización
        $demandas = DemandaHidricaM3::getCombinedDemandaHidrica()
            ->whereIn('nombre', $canales)  // Filtrar por los nombres de los canales
            ->groupBy(['anho', 'mes'])
            ->map(function ($group) {
                return [
                    'anho' => $group->first()->anho,
                    'mes' => $group->first()->mes,
                    'demanda_hidrica_bruta_m3' => $group->sum('demanda_hidrica_bruta_m3'),
                ];
            });

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima['anho'],
                'mes' => $nombresMeses[$demandaMaxima['mes']], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima['anho'],
                'mes' => $nombresMeses[$demandaMinima['mes']], // Convertir número de mes a nombre
            ],
        ]);
    }

    public function getRioDemanda(Request $request)
    {
        $seccionRioId = $request->input('rio'); // Obtener el seccionrio_id del request

        // Obtener los canales asociados a la sección del río
        $canales = DB::table('canals')
            ->where('canals.seccionrio_id', $seccionRioId) // Filtrar por seccionrio_id
            ->pluck('canals.nombre');

        // Obtener todas las demandas de las tres tablas para los canales de la sección del río
        $demandas = DemandaHidricaM3::getCombinedDemandaHidrica()
            ->whereIn('nombre', $canales)  // Filtrar por los nombres de los canales
            ->groupBy(['anho', 'mes'])
            ->map(function ($group) {
                return [
                    'anho' => $group->first()->anho,
                    'mes' => $group->first()->mes,
                    'demanda_hidrica_bruta_m3' => $group->sum('demanda_hidrica_bruta_m3'),
                ];
            });

        // Calcular el promedio, demanda máxima y mínima
        $promedio = $demandas->avg('demanda_hidrica_bruta_m3');
        $maxima = $demandas->max('demanda_hidrica_bruta_m3');
        $minima = $demandas->min('demanda_hidrica_bruta_m3');

        // Encontrar las filas correspondientes a la demanda máxima y mínima
        $demandaMaxima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $maxima);
        $demandaMinima = $demandas->firstWhere('demanda_hidrica_bruta_m3', $minima);

        // Arreglo de nombres de meses
        $nombresMeses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        // Devolver los resultados en formato JSON
        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'anho' => $demandaMaxima['anho'],
                'mes' => $nombresMeses[$demandaMaxima['mes']], // Convertir número de mes a nombre
            ],
            'minima' => [
                'valor' => $minima,
                'anho' => $demandaMinima['anho'],
                'mes' => $nombresMeses[$demandaMinima['mes']], // Convertir número de mes a nombre
            ],
        ]);
    }
}
