<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class GeojsonController extends Controller
{
    public function getGeojson(Request $request)
    {
        $path = public_path('assets/poligonos.geojson'); // Ruta al archivo consolidado

        if (!File::exists($path)) {
            return response()->json(['error' => 'File not found'], 404);
        }

        // Cargar el archivo GeoJSON
        $geojson = json_decode(File::get($path), true);

        $canalId = $request->input('canal_id');
        $comunaId = $request->input('comuna_id');
        $provinciaId = $request->input('provincia_id'); // Añadir filtro de provincia

        // Filtrar por canal_id, ID_COMUNA o provincia_id si existen
        if ($canalId || $comunaId || $provinciaId) {
            $filteredFeatures = array_filter($geojson['features'], function ($feature) use ($canalId, $comunaId, $provinciaId) {
                $canalMatch = true;
                $comunaMatch = true;
                $provinciaMatch = true;

                if ($canalId) {
                    $canalMatch = isset($feature['properties']['ID_CANAL']) && (int)$feature['properties']['ID_CANAL'] == (int)$canalId;
                }

                if ($comunaId) {
                    $comunaMatch = isset($feature['properties']['ID_COMUNA']) && (int)$feature['properties']['ID_COMUNA'] == (int)$comunaId;
                }

                // Filtrar por provincia a través de la comuna
                if ($provinciaId) {
                    $provinciaMatch = isset($feature['properties']['ID_COMUNA']) && $this->comunaBelongsToProvincia($feature['properties']['ID_COMUNA'], $provinciaId);
                }

                return $canalMatch && $comunaMatch && $provinciaMatch;
            });

            $geojson['features'] = array_values($filteredFeatures);
        }

        // Procesar geometrías (Polygon y MultiPolygon)
        foreach ($geojson['features'] as &$feature) {
            $geometryType = $feature['geometry']['type'];

            if ($geometryType === 'Polygon') {
                // Manejar Polygons
                $coordinates = $feature['geometry']['coordinates'];
                // Validaciones adicionales si son necesarias
            } elseif ($geometryType === 'MultiPolygon') {
                // Manejar MultiPolygons
                $coordinates = $feature['geometry']['coordinates'];
                // Validaciones adicionales si son necesarias
            }
        }

        // Paginación
        $page = $request->input('page', 1); // Página solicitada, por defecto 1
        $perPage = 1000; // Número de registros por página
        $total = count($geojson['features']);
        $totalPages = ceil($total / $perPage);

        // Definir rango para la paginación
        $start = ($page - 1) * $perPage;
        $pagedFeatures = array_slice($geojson['features'], $start, $perPage);

        // Preparar la respuesta con los datos paginados
        $pagedGeojson = [
            'type' => 'FeatureCollection',
            'features' => $pagedFeatures,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => $totalPages,
                'per_page' => $perPage,
                'total_features' => $total
            ]
        ];

        // Devolver la respuesta con los datos paginados
        return response()->json($pagedGeojson, 200);
    }

    // Función que verifica si una comuna pertenece a una provincia
    private function comunaBelongsToProvincia($comunaId, $provinciaId)
    {
        // Aquí debes implementar la lógica para verificar la relación
        // entre la comuna y la provincia. Puede ser desde la base de datos o un array estático.

        // Ejemplo con una estructura básica de provincias y comunas
        $provinciaToComunas = [
            '41' => ['4101', '4102', '4103', '4104', '4105', '4106'], // Provincia 41 (Elqui)
            '42' => ['4201', '4202', '4203', '4204'],                 // Provincia 42 (Choapa)
            '43' => ['4301', '4302', '4303', '4304', '4305'],         // Provincia 43 (Limarí)
        ];

        // Comprobar si la comuna está dentro de las comunas de la provincia
        return isset($provinciaToComunas[$provinciaId]) && in_array((string)$comunaId, $provinciaToComunas[$provinciaId]);
    }
}
