<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\DhElqui;
use App\Models\DhLimari;
use App\Models\DhChoapa;
use App\Models\OfertaElquiMpiLr;
use App\Models\OfertaLimariMpiLr;
use App\Models\OfertaChoapaMpiLr;
use App\Models\Canal;

class HydricDemandMpiLrController extends Controller
{
    public function getHydricDemandRWS(Request $request)
    {
        try {
            $canal_id = $request->input('canal_id');

            $canal = Canal::find($canal_id);

            if (!$canal) {
                return response()->json(['error' => 'Canal no encontrado'], 404);
            }

            // Obtener demandas, intentando en el siguiente orden: DhElqui, DhLimari, DhChoapa
            $demanda_historica = $this->getDemandaHistorica($canal_id);
            $demanda_proyectada = $this->getDemandaProyectada($canal_id);

            // Obtener ofertas agrupadas por mes y año
            $ofertas = $this->getOfertas($canal_id);

            // Procesar demandas históricas y proyectadas
            $resultados_historicos = $this->procesarDemandas($demanda_historica, $ofertas, $canal_id);
            $resultados_proyectados = $this->procesarDemandas($demanda_proyectada, $ofertas, $canal_id);

            $resultados = array_merge($resultados_historicos, $resultados_proyectados);

            // Agrupar por período y eliminar duplicados
            $agrupadoPorPeriodo = collect($resultados)->groupBy('periodo');
            $resultadosFinales = [];

            foreach ($agrupadoPorPeriodo as $periodo => $datos) {
                $uniqueData = collect($datos)->unique(function($item) {
                    return $item['canal'] . '-' . $item['temporada'];
                });

                $uniqueData = $uniqueData->sortByDesc('rws')->values()->toArray();
                $total = count($uniqueData);

                foreach ($uniqueData as $index => &$dato) {
                    $dato['weibull'] = ($index + 1) / ($total + 1);
                }
                $resultadosFinales = array_merge($resultadosFinales, $uniqueData);
            }

            return response()->json($resultadosFinales);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function getDemandaHistorica($canal_id)
    {
        $demanda = DhElqui::where('canal_id', $canal_id)
            ->where('anho', '<', 2024)
            ->get();

        if ($demanda->isEmpty()) {
            $demanda = DhLimari::where('canal_id', $canal_id)
                ->where('anho', '<', 2024)
                ->get();
        }

        if ($demanda->isEmpty()) {
            $demanda = DhChoapa::where('canal_id', $canal_id)
                ->where('anho', '<', 2024)
                ->get();
        }

        return $demanda;
    }

    private function getDemandaProyectada($canal_id)
    {
        $demanda = DhElqui::where('canal_id', $canal_id)
            ->where('anho', '>=', 2024)
            ->get();

        if ($demanda->isEmpty()) {
            $demanda = DhLimari::where('canal_id', $canal_id)
                ->where('anho', '>=', 2024)
                ->get();
        }

        if ($demanda->isEmpty()) {
            $demanda = DhChoapa::where('canal_id', $canal_id)
                ->where('anho', '>=', 2024)
                ->get();
        }

        return $demanda;
    }

    private function getOfertas($canal_id)
    {
        $ofertas = OfertaElquiMpiLr::where('canal_id', $canal_id)->get()->groupBy(function ($item) {
            return \Carbon\Carbon::parse($item->fecha)->format('Y-m');
        });

        if ($ofertas->isEmpty()) {
            $ofertas = OfertaLimariMpiLr::where('canal_id', $canal_id)->get()->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->fecha)->format('Y-m');
            });
        }

        if ($ofertas->isEmpty()) {
            $ofertas = OfertaChoapaMpiLr::where('canal_id', $canal_id)->get()->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->fecha)->format('Y-m');
            });
        }

        return $ofertas;
    }

    private function procesarDemandas($demandas, $ofertas, $canal_id)
    {
        $resultados = [];
        $temporadaDemanda = [];

        foreach ($demandas as $demanda) {
            $temporada = $this->asignarTemporada($demanda->anho, $demanda->mes);

            if (!isset($temporadaDemanda[$temporada])) {
                $temporadaDemanda[$temporada] = 0;
            }
            $demanda_hidrica_mes = floatval(str_replace(',', '.', $demanda->demanda_hidrica));
            $temporadaDemanda[$temporada] += $demanda_hidrica_mes;
        }

        foreach ($temporadaDemanda as $temporada => $demanda_hidrica_total) {
            $oferta = $this->getOfertaForSeason($temporada, $ofertas, $canal_id);

            if ($oferta) {
                $oferta_hidrica = floatval($oferta->ofertahidrica);

                if ($demanda_hidrica_total > 0) {
                    $rws = $oferta_hidrica / $demanda_hidrica_total;

                    $periodo = $this->asignarPeriodo(explode('-', $temporada)[0]);
                    $resultados[] = [
                        'canal' => $demandas[0]->canal,
                        'temporada' => $temporada,
                        'demanda_hidrica' => $demanda_hidrica_total,
                        'oferta_hidrica' => $oferta_hidrica,
                        'rws' => $rws,
                        'periodo' => $periodo,
                    ];

                    if ($periodo !== 'Histórico 2000-2024') {
                        $resultados[] = [
                            'canal' => $demandas[0]->canal,
                            'temporada' => $temporada,
                            'demanda_hidrica' => $demanda_hidrica_total,
                            'oferta_hidrica' => $oferta_hidrica,
                            'rws' => $rws,
                            'periodo' => 'Proyectado completo 2025-2094',
                        ];
                    }
                }
            }
        }

        return $resultados;
    }

    private function asignarTemporada($anho, $mes)
    {
        if ($mes >= 9) {
            return "{$anho}-" . ($anho + 1);
        } else {
            return ($anho - 1) . "-{$anho}";
        }
    }

    private function asignarPeriodo($anho)
    {
        if ($anho < 2024) {
            return 'Histórico 2000-2024';
        } elseif ($anho >= 2024 && $anho < 2045) {
            return 'Corto plazo 2025-2044';
        } elseif ($anho >= 2045 && $anho < 2070) {
            return 'Mediano plazo 2025-2069';
        } elseif ($anho >= 2025 && $anho <= 2094) {
            return 'Largo plazo 2025-2094';
        } else {
            return 'Proyectado completo 2025-2094';
        }
    }

    private function getOfertaForSeason($season, $ofertas, $canal_id)
    {
        $seasonParts = explode('-', $season);
        $seasonStartYear = intval($seasonParts[0]);
        $seasonEndYear = intval($seasonParts[1]);

        $totalOfertaHidrica = 0;
        $usedMonths = [];

        for ($year = $seasonStartYear; $year <= $seasonEndYear; $year++) {
            $months = $year == $seasonStartYear ? range(9, 12) : ($year == $seasonEndYear ? range(1, 8) : range(1, 12));

            foreach ($months as $month) {
                if (in_array($month, $usedMonths)) {
                    continue; // Evitar duplicación si ya se ha usado este mes
                }

                $monthFormatted = str_pad($month, 2, '0', STR_PAD_LEFT);
                $key = "{$year}-{$monthFormatted}";

                if (isset($ofertas[$key])) {
                    foreach ($ofertas[$key] as $oferta) {
                        if ($oferta->canal_id == $canal_id) {
                            $oferta_hidrica_mes = floatval(str_replace(',', '.', $oferta->ofertahidrica));
                            $totalOfertaHidrica += $oferta_hidrica_mes;
                            $usedMonths[] = $month; // Marcar el mes como usado para evitar duplicados
                            break; // Solo utilizamos la primera oferta encontrada para este mes
                        }
                    }
                }
            }
        }

        return (object)['ofertahidrica' => $totalOfertaHidrica];
    }
}
