<?php

namespace App\Http\Controllers;

use App\Models\Canal;
use App\Models\EficienciaRiegoSistema;
use App\Models\Kc;
use App\Models\OfertaSeccRioElquiCsiro;
use App\Models\OfertaSeccRioLimarCsiro;
use App\Models\OfertaSeccRioChoapaCsiro;
use App\Models\Etomensual;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class IndicadorOrphanopoulusController extends Controller
{
    public function calcularIndicadorOrphanopoulus($canal_id, $periodo, $superficie, $grupo_id, $sistema_riego, $daa, Request $request)
{
    // Verificación y carga inicial
    if (!$grupo_id) {
        return response()->json(['error' => 'Grupo de cultivo no especificado'], 400);
    }

    $canal = Canal::with(['sectorhidrico', 'seccionrio', 'comuna', 'subsubcuenca'])->find($canal_id);
    if (!$canal) {
        return response()->json(['error' => 'Canal no encontrado'], 404);
    }

    $eficienciaSistema = EficienciaRiegoSistema::where('sistema', $sistema_riego)->first();
    if (!$eficienciaSistema) {
        return response()->json(['error' => 'Eficiencia del sistema de riego no encontrada'], 404);
    }

    $eficienciaConduccion = $request->query('eficiencia_conduccion') / 100;
    if ($eficienciaConduccion <= 0 || $eficienciaConduccion > 1) {
        return response()->json(['error' => 'Eficiencia de conducción debe estar entre 1% y 100%'], 400);
    }

    // Nuevo: capturar el valor de oferta_percentil_85 del frontend
    $ofertaPercentil85 = $request->query('oferta_percentil_85', 0);
    $ofertaMensualProyectada = $ofertaPercentil85 / 12; // Oferta mensual para años proyectados

    // Continuar con el cálculo
    list($startYear, $endYear) = $this->getSeasonPeriod($this->determinarProvincia($canal->comuna_id), $periodo);
    $promedioDemandaMensual = $this->calcularPromedioDemandaMensualPorMes($canal->sectorhidrico_id, $superficie, $grupo_id, $eficienciaSistema->eficiencia);
    $aniosSatisfactorios = 0;
    $totalAnios = $endYear - $startYear + 1;
    $resultadosTemporada = [];

    for ($year = $startYear; $year <= $endYear; $year++) {
        $satisfaccionMensual = [];
        $seasonMonths = $this->calculateSeasonMonths($this->determinarProvincia($canal->comuna_id), $year);

        foreach ($seasonMonths as $period) {
            $currentYear = $period['year'];
            $month = $period['month'];

            if ($year >= 2024) {
                // Usar la oferta mensual proyectada para años proyectados
                $demandaMensual = $promedioDemandaMensual[$month];
                $ofertaMensual = $ofertaMensualProyectada;

            } else {
                // Cálculo de oferta y demanda para años históricos
                $oferta = $this->obtenerOfertaMensual($canal, $currentYear, $month);
                $kc = Kc::where('grupo_id', $grupo_id)
                    ->where('sectorhidrico_id', $canal->sectorhidrico_id)
                    ->where('anho', $currentYear)
                    ->where('mes', $month)
                    ->first();
                $eto = Etomensual::where('sectorhidrico_id', $canal->sectorhidrico_id)
                    ->where('anho', $currentYear)
                    ->where('mes', $month)
                    ->first();

                $kcValue = $kc->kc ?? $this->calcularPromedioKc($grupo_id, $canal->sectorhidrico_id);
                $etoValue = $eto->eto ?? $this->calcularPromedioEto($canal->sectorhidrico_id);

                $demandaMensual = ($superficie * $kcValue * $etoValue * 10) / $eficienciaSistema->eficiencia;
                $ofertaMensual = $oferta ? $oferta->ofertahidrica * $daa * $eficienciaConduccion : 0;

            }

            $indiceOrphanopoulus = $demandaMensual > 0 ? $ofertaMensual / $demandaMensual : 0;
            $satisfaccionMensual[$month] = $indiceOrphanopoulus;
        }

        // Calculo de meses satisfactorios
        $mesesSatisfaccion90 = count(array_filter($satisfaccionMensual, fn($indice) => $indice >= 0.9));
        $mesesSatisfaccion80 = count(array_filter($satisfaccionMensual, fn($indice) => $indice >= 0.8 && $indice < 0.9));

        $satisfactorio = ($mesesSatisfaccion90 >= 11 || ($mesesSatisfaccion90 == 10 && $mesesSatisfaccion80 >= 1));
        $resultadosTemporada[$year] = [
            'satisfactorio' => $satisfactorio,
            'meses_90' => $mesesSatisfaccion90,
            'meses_80' => $mesesSatisfaccion80,
        ];


        if ($satisfactorio) {
            $aniosSatisfactorios++;
        }
    }

    $indiceSatisfaccionRiego = $aniosSatisfactorios / $totalAnios;
    $clasificacionFinal = $this->clasificarIndiceSatisfaccion($indiceSatisfaccionRiego);

    return response()->json([
        'canal' => [
            'id' => $canal->id,
            'nombre' => $canal->nombre,
            'daatotales' => $canal->daatotales,
            'sectorhidrico_id' => $canal->sectorhidrico_id,
            'seccionrio_id' => $canal->seccionrio_id,
            'comuna_id' => $canal->comuna_id,
            'subsubcuenca_id' => $canal->subsubcuenca_id,
        ],
        'resultados_temporada' => $resultadosTemporada,
        'indice_satisfaccion_riego' => $indiceSatisfaccionRiego,
        'clasificacion_final' => $clasificacionFinal,
    ]);
}



    private function calcularPromedioDemandaMensualPorMes($sectorhidrico_id, $superficie, $grupo_id, $eficienciaSistema)
    {
        $promedios = [];

        for ($mes = 1; $mes <= 12; $mes++) {
            $demandaMensualTotal = 0;
            $count = 0;

            for ($year = 2000; $year <= 2023; $year++) {
                $kc = Kc::where('grupo_id', $grupo_id)
                    ->where('sectorhidrico_id', $sectorhidrico_id)
                    ->where('anho', $year)
                    ->where('mes', $mes)
                    ->first();

                $eto = Etomensual::where('sectorhidrico_id', $sectorhidrico_id)
                    ->where('anho', $year)
                    ->where('mes', $mes)
                    ->first();

                if ($kc && $eto) {
                    $demandaMensual = ($superficie * $kc->kc * $eto->eto * 10) / $eficienciaSistema;
                    $demandaMensualTotal += $demandaMensual;
                    $count++;
                }
            }

            $promedios[$mes] = $count > 0 ? $demandaMensualTotal / $count : 0;

        }

        return $promedios;
    }



    private function obtenerOfertaMensual($canal, $year, $month)
    {

        $oferta = OfertaSeccRioElquiCsiro::where('seccionrio_id', $canal->seccionrio_id)
            ->where('fecha', $year)
            ->where('mes', $month)
            ->first();

        if ($oferta) {
        } else {
            $oferta = OfertaSeccRioLimarCsiro::where('seccionrio_id', $canal->seccionrio_id)
                ->where('fecha', $year)
                ->where('mes', $month)
                ->first();
            if ($oferta) {
            }
        }

        if (!$oferta) {
            $oferta = OfertaSeccRioChoapaCsiro::where('seccionrio_id', $canal->seccionrio_id)
                ->where('fecha', $year)
                ->where('mes', $month)
                ->first();
            if ($oferta) {
            } else {
                Log::warning("No se encontró oferta para la sección de río " . $canal->seccionrio_id . " en el año $year y mes $month");
            }
        }

        return $oferta;
    }

    private function clasificarIndiceSatisfaccion($indice)
    {
        if ($indice >= 0.9) {
            return '5 (muy bueno)';
        } elseif ($indice >= 0.8) {
            return '4 (bueno)';
        } elseif ($indice >= 0.7) {
            return '3 (regular)';
        } elseif ($indice >= 0.6) {
            return '2 (malo)';
        } else {
            return '1 (muy malo)';
        }
    }

    private function determinarProvincia($comuna_id)
    {
        $provinciaCode = substr($comuna_id, 0, 2);
        switch ($provinciaCode) {
            case '41': return 'Elqui';
            case '42': return 'Limarí';
            case '43': return 'Choapa';
            default:
                return null;
        }
    }

    private function getSeasonPeriod($provincia, $periodo)
    {
        $startYear = $this->getYearStart($periodo);
        $endYear = $this->getYearEnd($periodo);
        return [$startYear, $endYear];
    }

    private function calculateSeasonMonths($provincia, $startYear)
    {
        $seasonStartMonth = $this->getSeasonStartMonth($provincia);

        $seasonMonths = [];
        for ($month = $seasonStartMonth; $month <= 12; $seasonMonths[] = ['year' => $startYear, 'month' => $month++]);
        for ($month = 1; $month < $seasonStartMonth; $seasonMonths[] = ['year' => $startYear + 1, 'month' => $month++] );

        return $seasonMonths;
    }

    private function calcularPromedioKc($grupo_id, $sectorhidrico_id)
    {
        return Kc::where('grupo_id', $grupo_id)
            ->where('sectorhidrico_id', $sectorhidrico_id)
            ->avg('kc');
    }

    private function calcularPromedioEto($sectorhidrico_id)
    {
        return Etomensual::where('sectorhidrico_id', $sectorhidrico_id)
            ->avg('eto');
    }

    private function getSeasonStartMonth($provincia)
    {
        return ($provincia === 'Elqui') ? 9 : 5;
    }

    private function getYearStart($periodo)
    {
        if ($periodo === 'Historico 2000-2024') {
            return 2000;
        } elseif ($periodo === 'Corto plazo 2025-2044') {
            return 2025;
        } elseif ($periodo === 'Mediano plazo 2025-2069') {
            return 2045;
        } elseif ($periodo === 'Largo plazo 2025-2094') {
            return 2070;
        } else {
            return 2000;
        }
    }

    private function getYearEnd($periodo)
    {
        if ($periodo === 'Historico 2000-2024') {
            return 2023;
        } elseif ($periodo === 'Corto plazo 2025-2044') {
            return 2044;
        } elseif ($periodo === 'Mediano plazo 2025-2069') {
            return 2069;
        } elseif ($periodo === 'Largo plazo 2025-2094') {
            return 2094;
        } else {
            return 2024;
        }
    }
}
