<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Canal;
use App\Models\DhElqui;
use App\Models\DhLimari;
use App\Models\DhChoapa;
use App\Models\OfertaElquiIpsl;
use App\Models\OfertaLimariIpsl;
use App\Models\OfertaChoapaIpsl;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OrphanopoulusIndicatorIpslController extends Controller
{
    public function calculateOrphanopoulusIndicator(Request $request)
    {
        try {
            $canal_id = $request->input('canal_id');
            $canal = Canal::find($canal_id);

            if (!$canal) {
                return response()->json(['error' => 'Canal no encontrado'], 404);
            }

            // Buscar demandas en las tablas en el orden de prioridad: DhElqui, DhLimari, DhChoapa.
            $demandas = DhElqui::where('canal_id', $canal_id)->get();

            if ($demandas->isEmpty()) {
                $demandas = DhLimari::where('canal_id', $canal_id)->get();
            }

            if ($demandas->isEmpty()) {
                $demandas = DhChoapa::where('canal_id', $canal_id)->get();
            }

            // Si no hay demandas en ninguna de las tablas, retornar un error.
            if ($demandas->isEmpty()) {
                return response()->json(['error' => 'No se encontraron demandas para el canal en ninguna de las tablas.'], 404);
            }

            // Intentar buscar ofertas en OfertaElquiIpsl, y luego en OfertaLimariIpsl y OfertaChoapaIpsl si no hay resultados.
            $ofertas = OfertaElquiIpsl::where('canal_id', $canal_id)->get()->groupBy(function ($date) {
                return \Carbon\Carbon::parse($date->fecha)->format('Y-m');
            });

            if ($ofertas->isEmpty()) {
                $ofertas = OfertaLimariIpsl::where('canal_id', $canal_id)->get()->groupBy(function ($date) {
                    return \Carbon\Carbon::parse($date->fecha)->format('Y-m');
                });
            }

            if ($ofertas->isEmpty()) {
                $ofertas = OfertaChoapaIpsl::where('canal_id', $canal_id)->get()->groupBy(function ($date) {
                    return \Carbon\Carbon::parse($date->fecha)->format('Y-m');
                });
            }

            // Si no se encuentran ofertas en ninguna tabla, retornar un error.
            if ($ofertas->isEmpty()) {
                return response()->json(['error' => 'No se encontraron ofertas para el canal en ninguna de las tablas.'], 404);
            }

            // Calcular el RWS para cada demanda.
            $resultadosMensuales = [];
            $resultadosTemporadas = [];
            foreach ($demandas as $demanda) {
                $oferta = $this->getOfertaForMonth($demanda->anho, $demanda->mes, $ofertas);

                if ($oferta) {
                    $demanda_hidrica = floatval(str_replace(',', '.', $demanda->demanda_hidrica));
                    $oferta_hidrica = floatval($oferta->ofertahidrica);

                    if ($demanda_hidrica > 0) {
                        $rws = $oferta_hidrica / $demanda_hidrica;
                        $resultadosMensuales[] = [
                            'canal' => $canal->nombre,
                            'anho' => $demanda->anho,
                            'mes' => $demanda->mes,
                            'demanda_hidrica' => $demanda_hidrica,
                            'oferta_hidrica' => $oferta_hidrica,
                            'rws' => $rws,
                        ];

                        $temporada = $this->getSeason($demanda->anho, $demanda->mes);
                        $resultadosTemporadas[$temporada][] = [
                            'mes' => $demanda->mes,
                            'rws' => $rws,
                        ];
                    } else {
                        Log::warning("Demanda hídrica es cero para el canal {$canal->nombre} en el mes {$demanda->mes} del año {$demanda->anho}");
                    }
                }
            }

            // Calcular los indicadores por temporada y por periodo.
            $indicadorTemporadas = $this->calcularIndicadorPorTemporadas($resultadosTemporadas);
            $indicadorPeriodos = $this->calcularIndicadorPorPeriodos($indicadorTemporadas);

            return response()->json([
                'mensual' => $resultadosMensuales,
                'temporadas' => $indicadorTemporadas,
                'periodos' => $indicadorPeriodos
            ]);
        } catch (\Exception $e) {
            Log::error("Error en el proceso: " . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function getSeason($anho, $mes)
    {
        if ($mes >= 9) {
            return $anho . '-' . ($anho + 1);
        }
        return ($anho - 1) . '-' . $anho;
    }

    private function calcularIndicadorPorTemporadas($resultadosTemporadas)
    {
        $indicadorTemporadas = [];

        foreach ($resultadosTemporadas as $temporada => $meses) {
            $satisfaccion90 = count(array_filter($meses, function ($mes) {
                return $mes['rws'] >= 0.9;
            }));

            $satisfaccion80 = count(array_filter($meses, function ($mes) {
                return $mes['rws'] >= 0.8;
            }));

            $satisfactoria = $satisfaccion90 >= (count($meses) - 1) && $satisfaccion80 >= 1;

            $indice = $satisfactoria ? 1 : 0;

            $indicadorTemporadas[$temporada] = [
                'indice' => $indice,
                'descripcion' => $this->obtenerDescripcionIndicador($indice),
                'detalles' => $meses
            ];
        }

        return $indicadorTemporadas;
    }

    private function calcularIndicadorPorPeriodos($indicadorTemporadas)
{
    $periodos = [
        'Histórico' => ['inicio' => 2000, 'fin' => 2024],
        'Corto plazo' => ['inicio' => 2025, 'fin' => 2044],
        'Mediano plazo' => ['inicio' => 2025, 'fin' => 2069],  // Ajuste aquí
        'Largo plazo' => ['inicio' => 2025, 'fin' => 2094]      // Ajuste aquí
    ];

    $indicadorPeriodos = [];

    foreach ($periodos as $periodo => $rango) {
        $totalTemporadas = 0;
        $temporadasSatisfactorias = 0;

        foreach ($indicadorTemporadas as $temporada => $detalles) {
            $temporadaInicio = (int)substr($temporada, 0, 4);
            if ($temporadaInicio >= $rango['inicio'] && $temporadaInicio <= $rango['fin']) {
                $totalTemporadas++;
                if ($detalles['indice'] === 1) {
                    $temporadasSatisfactorias++;
                }
            }
        }

        if ($totalTemporadas > 0) {
            $indice = $temporadasSatisfactorias / $totalTemporadas;
            $descripcion = $this->obtenerDescripcionIndicador($indice);
            $indicadorPeriodos[$periodo] = [
                'indice' => $indice,
                'descripcion' => $descripcion
            ];
        }
    }

    return $indicadorPeriodos;
}


    private function obtenerDescripcionIndicador($indice)
    {
        if ($indice >= 0.9) {
            return '5 (Muy bueno)';
        } elseif ($indice >= 0.85) {
            return '4 (Bueno)';
        } elseif ($indice >= 0.75) {
            return '3 (Regular)';
        } elseif ($indice >= 0.6) {
            return '2 (Malo)';
        } else {
            return '1 (Muy malo)';
        }
    }

    private function getOfertaForMonth($anho, $mes, $ofertas)
    {
        $key = "{$anho}-" . str_pad($mes, 2, '0', STR_PAD_LEFT);

        $totalOfertaHidrica = 0;

        if (isset($ofertas[$key])) {
            foreach ($ofertas[$key] as $oferta) {
                $totalOfertaHidrica += floatval(str_replace(',', '.', $oferta->ofertahidrica));
            }
        }

        return (object)['ofertahidrica' => $totalOfertaHidrica];
    }
}
