<?php

namespace App\Http\Controllers;

use App\Models\SuperficieActiva;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SuperficieActivaController extends Controller
{
    public function getSuperficie(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $temporada = $request->input('temporada');

        // Seleccionar la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provincia);

        // Superficie Promedio
        $promedio = DB::table($table)
            ->where('canal_v2', 'like', '%' . $canal . '%')
            ->where('temporada', $temporada)
            ->avg('area_ha');

        // Superficie Máxima
        $maxima = DB::table($table)
            ->where('canal_v2', 'like', '%' . $canal . '%')
            ->where('temporada', $temporada)
            ->max('area_ha');

        // Superficie Mínima
        $minima = DB::table($table)
            ->where('canal_v2', 'like', '%' . $canal . '%')
            ->where('temporada', $temporada)
            ->min('area_ha');

        return response()->json([
            'promedio' => $promedio,
            'maxima' => $maxima,
            'minima' => $minima,
        ]);
    }

    // Obtener la superficie por año de un canal
    public function getCanalSuperficie(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');

        // Seleccionar la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provincia);

        // Obtener todas las superficies para el canal seleccionado
        $superficies = DB::table($table)
            ->where('canal_v2', 'like', '%' . $canal . '%')
            ->get(['temporada', 'area_ha']);

        // Calcular el promedio, superficie máxima y mínima
        $promedio = $superficies->avg('area_ha');
        $maxima = $superficies->max('area_ha');
        $minima = $superficies->min('area_ha');

        // Encontrar las temporadas correspondientes a la superficie máxima y mínima
        $temporadaMaxima = $superficies->firstWhere('area_ha', $maxima)->temporada ?? null;
        $temporadaMinima = $superficies->firstWhere('area_ha', $minima)->temporada ?? null;

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'temporada' => $temporadaMaxima,
            ],
            'minima' => [
                'valor' => $minima,
                'temporada' => $temporadaMinima,
            ],
        ]);
    }
    // Obtener las superficies de todos los años de un canal
    public function getSuperficieByYear(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');

        // Seleccionar la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provincia);

        // Obtener las superficies desde el año 2000
        $superficie = DB::table($table)
            ->where('canal_v2', 'like', '%' . $canal . '%')
            ->select('temporada', 'area_ha')
            ->orderBy('temporada', 'asc')
            ->get();

        return response()->json($superficie);
    }

    // obtener la superficie de un unico año
    public function getSuperficieByTemporada(Request $request)
    {
        $provincia = $request->input('provincia');
        $canal = $request->input('canal');
        $temporada = $request->input('temporada');

        // Seleccionar la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provincia);

        // Obtener la superficie de un único año
        $superficie = DB::table($table)
            ->where('canal_v2', 'like', '%' . $canal . '%')
            ->where('temporada', $temporada)
            ->get();

        return response()->json($superficie);
    }

    public function getSuperficiePorTemporadaYCultivo(Request $request)
{
    $provincia = $request->input('provincia');
    $canalId = $request->input('canal_id');
    $rangeStart = $request->input('rangeStart');
    $rangeEnd = $request->input('rangeEnd');

    // Obtener el nombre del canal usando el canal_id proporcionado
    $canalNombre = DB::table('canals')
        ->where('id', $canalId)
        ->value('nombre');


    if (!$canalNombre) {
        return response()->json([], 404);
    }

    // Seleccionar la tabla según el primer dígito del canal ID
    if (str_starts_with($canalId, '1')) {
        $table = 'superficie_activa_elqui_por_temporada';
    } elseif (str_starts_with($canalId, '2')) {
        $table = 'superficie_activa_limari_por_temporada';
    } elseif (str_starts_with($canalId, '3')) {
        $table = 'superficie_activa_choapa_por_temporada';
    } else {
        return response()->json([], 404);
    }


    // Realizar la consulta a la tabla de superficies
    $query = DB::table($table)
        ->where('canal_v2', '=', $canalNombre)
        ->whereBetween('temporada', [$rangeStart, $rangeEnd])
        ->select(
            'temporada',
            DB::raw('SUM(area_ha) as superficie_total'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
        )
        ->groupBy('temporada')
        ->orderBy('temporada', 'asc');


    $superficiesPorTemporada = $query->get();

    if ($superficiesPorTemporada->isEmpty()) {
    } else {
    }

    // Formatear resultados antes de enviarlos
    $result = [];
    foreach ($superficiesPorTemporada as $superficie) {
        $result[] = [
            'temporada' => $superficie->temporada,
            'superficie_total' => $superficie->superficie_total,
            'caduco' => $superficie->caduco,
            'pradera' => $superficie->pradera,
            'ciclo_corto' => $superficie->ciclo_corto,
            'persistente' => $superficie->persistente,
        ];
    }

    return response()->json($result);
}



    public function getProvinciaSuperficie(Request $request)
    {
        $provinciaId = $request->input('provincia');

        // Seleccionar la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provinciaId);

        // Obtener todas las comunas de la provincia seleccionada
        $comunas = DB::table('comunas')
            ->where('provincia_id', $provinciaId)
            ->pluck('id');

        // Obtener todos los canales de las comunas seleccionadas
        $canales = DB::table('canals')
            ->whereIn('comuna_id', $comunas)
            ->pluck('nombre');

        // Obtener todas las superficies para los canales de la provincia
        $superficies = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->get(['temporada', 'area_ha']);

        // Calcular el promedio, superficie máxima y mínima
        $promedio = $superficies->avg('area_ha');
        $maxima = $superficies->max('area_ha');
        $minima = $superficies->min('area_ha');

        // Encontrar las temporadas correspondientes a la superficie máxima y mínima
        $temporadaMaxima = $superficies->firstWhere('area_ha', $maxima)->temporada ?? null;
        $temporadaMinima = $superficies->firstWhere('area_ha', $minima)->temporada ?? null;

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'temporada' => $temporadaMaxima,
            ],
            'minima' => [
                'valor' => $minima,
                'temporada' => $temporadaMinima,
            ],
        ]);
    }

    public function getSuperficiePorTemporadaYCultivoProvincia(Request $request)
    {
        $provinciaId = $request->input('provincia');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provinciaId);

        // Obtener todas las comunas de la provincia seleccionada
        $comunas = DB::table('comunas')
            ->where('provincia_id', $provinciaId)
            ->pluck('id');

        // Obtener todos los canales de las comunas seleccionadas
        $canales = DB::table('canals')
            ->whereIn('comuna_id', $comunas)
            ->pluck('nombre');

        // Obtener la superficie total y por tipo de cultivo en el rango de años seleccionado
        $superficiesPorTemporada = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->whereBetween('temporada', [$rangeStart, $rangeEnd])
            ->select(
                'temporada',
                DB::raw('SUM(area_ha) as superficie_total'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
            )
            ->groupBy('temporada')
            ->orderBy('temporada', 'asc')
            ->get();

        $result = [];
        foreach ($superficiesPorTemporada as $superficie) {
            $result[] = [
                'temporada' => $superficie->temporada,
                'superficie_total' => $superficie->superficie_total,
                'caduco' => $superficie->caduco,
                'pradera' => $superficie->pradera,
                'ciclo_corto' => $superficie->ciclo_corto,
                'persistente' => $superficie->persistente,
            ];
        }

        return response()->json($result);
    }



    public function getComunaSuperficie(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');

        // Seleccionar la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provincia);

        // Obtener todos los canales de la comuna seleccionada
        $canales = DB::table('canals')
            ->where('comuna_id', $comunaId)
            ->pluck('nombre');

        // Obtener todas las superficies para los canales de la comuna
        $superficies = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->get(['temporada', 'area_ha']);

        // Calcular el promedio, superficie máxima y mínima
        $promedio = $superficies->avg('area_ha');
        $maxima = $superficies->max('area_ha');
        $minima = $superficies->min('area_ha');

        // Encontrar las temporadas correspondientes a la superficie máxima y mínima
        $temporadaMaxima = $superficies->firstWhere('area_ha', $maxima)->temporada ?? null;
        $temporadaMinima = $superficies->firstWhere('area_ha', $minima)->temporada ?? null;

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'temporada' => $temporadaMaxima,
            ],
            'minima' => [
                'valor' => $minima,
                'temporada' => $temporadaMinima,
            ],
        ]);
    }



    public function getSuperficiePorTemporadaYCultivoComuna(Request $request)
    {
        $provincia = $request->input('provincia');
        $comunaId = $request->input('comuna');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la provincia
        $table = SuperficieActiva::setTableForProvince($provincia);

        // Obtener todos los canales de la comuna seleccionada
        $canales = DB::table('Canals')
            ->where('comuna_id', $comunaId)
            ->pluck('nombre');

        // Obtener la superficie total y por tipo de cultivo en el rango de años seleccionado
        $superficiesPorTemporada = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->whereBetween('temporada', [$rangeStart, $rangeEnd])
            ->select(
                'temporada',
                DB::raw('SUM(area_ha) as superficie_total'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
            )
            ->groupBy('temporada')
            ->orderBy('temporada', 'asc')
            ->get();

        $result = [];
        foreach ($superficiesPorTemporada as $superficie) {
            $result[] = [
                'temporada' => $superficie->temporada,
                'superficie_total' => $superficie->superficie_total,
                'caduco' => $superficie->caduco,
                'pradera' => $superficie->pradera,
                'ciclo_corto' => $superficie->ciclo_corto,
                'persistente' => $superficie->persistente,
            ];
        }

        return response()->json($result);
    }

    public function getCuencaSuperficie(Request $request)
    {
        $cuencaId = $request->input('cuenca');

        // Obtener los canales asociados a la cuenca (a través de subcuencas y subsubcuencas)
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->join('subcuencas', 'subsubcuencas.subcuenca_id', '=', 'subcuencas.id')
            ->where('subcuencas.cuenca_id', $cuencaId)
            ->pluck('canals.nombre');

        // Obtener todas las superficies de las tres tablas para los canales de la cuenca
        $superficies = SuperficieActiva::getCombinedSuperficie()
            ->whereIn('canal_v2', $canales)  // Filtrar por los nombres de los canales
            ->groupBy('temporada')
            ->map(function ($group) {
                return [
                    'temporada' => $group->first()->temporada,
                    'area_ha' => $group->sum('area_ha'),
                ];
            });

        // Calcular el promedio, superficie máxima y mínima
        $promedio = $superficies->avg('area_ha');
        $maxima = $superficies->max('area_ha');
        $minima = $superficies->min('area_ha');

        // Encontrar las filas correspondientes a la superficie máxima y mínima
        $superficieMaxima = $superficies->firstWhere('area_ha', $maxima);
        $superficieMinima = $superficies->firstWhere('area_ha', $minima);

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'temporada' => $superficieMaxima['temporada'],
            ],
            'minima' => [
                'valor' => $minima,
                'temporada' => $superficieMinima['temporada'],
            ],
        ]);
    }


    public function getSubcuencaSuperficie(Request $request)
    {
        $subcuencaId = $request->input('subcuenca');

        // Obtener los canales asociados a la subcuenca (a través de subsubcuencas)
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->where('subsubcuencas.subcuenca_id', $subcuencaId)
            ->pluck('canals.nombre');

        // Obtener todas las superficies de las tres tablas para los canales de la subcuenca
        $superficies = SuperficieActiva::getCombinedSuperficie()
            ->whereIn('canal_v2', $canales)  // Filtrar por los nombres de los canales
            ->groupBy('temporada')
            ->map(function ($group) {
                return [
                    'temporada' => $group->first()->temporada,
                    'area_ha' => $group->sum('area_ha'),
                ];
            });

        // Calcular el promedio, superficie máxima y mínima
        $promedio = $superficies->avg('area_ha');
        $maxima = $superficies->max('area_ha');
        $minima = $superficies->min('area_ha');

        // Encontrar las filas correspondientes a la superficie máxima y mínima
        $superficieMaxima = $superficies->firstWhere('area_ha', $maxima);
        $superficieMinima = $superficies->firstWhere('area_ha', $minima);

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'temporada' => $superficieMaxima['temporada'],
            ],
            'minima' => [
                'valor' => $minima,
                'temporada' => $superficieMinima['temporada'],
            ],
        ]);
    }

    public function getSuperficiePorTemporadaYCultivoCuenca(Request $request)
    {
        $cuencaId = $request->input('cuenca');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la cuenca
        $table = SuperficieActiva::getCombinedSuperficieTemporadaYCultivo($cuencaId);

        // Obtener todos los canales de la cuenca seleccionada
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->join('subcuencas', 'subsubcuencas.subcuenca_id', '=', 'subcuencas.id')
            ->where('subcuencas.cuenca_id', $cuencaId)
            ->pluck('canals.nombre');

        // Obtener la superficie total y por tipo de cultivo en el rango de años seleccionado
        $superficiesPorTemporada = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->whereBetween('temporada', [$rangeStart, $rangeEnd])
            ->select(
                'temporada',
                DB::raw('SUM(area_ha) as superficie_total'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
            )
            ->groupBy('temporada')
            ->orderBy('temporada', 'asc')
            ->get();

        $result = [];
        foreach ($superficiesPorTemporada as $superficie) {
            $result[] = [
                'temporada' => $superficie->temporada,
                'superficie_total' => $superficie->superficie_total,
                'caduco' => $superficie->caduco,
                'pradera' => $superficie->pradera,
                'ciclo_corto' => $superficie->ciclo_corto,
                'persistente' => $superficie->persistente,
            ];
        }

        return response()->json($result);
    }


    public function getSuperficiePorTemporadaYCultivoSubcuenca(Request $request)
    {
        $subcuencaId = $request->input('subcuenca');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Selecciona la tabla según la subcuenca
        $cuencaId = DB::table('subcuencas')->where('id', $subcuencaId)->value('cuenca_id');
        $table = SuperficieActiva::getCombinedSuperficieTemporadaYCultivo($cuencaId);

        // Obtener todos los canales de la subcuenca seleccionada
        $canales = DB::table('canals')
            ->join('subsubcuencas', 'canals.subsubcuenca_id', '=', 'subsubcuencas.id')
            ->where('subsubcuencas.subcuenca_id', $subcuencaId)
            ->pluck('canals.nombre');

        // Obtener la superficie total y por tipo de cultivo en el rango de años seleccionado
        $superficiesPorTemporada = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->whereBetween('temporada', [$rangeStart, $rangeEnd])
            ->select(
                'temporada',
                DB::raw('SUM(area_ha) as superficie_total'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
            )
            ->groupBy('temporada')
            ->orderBy('temporada', 'asc')
            ->get();

        $result = [];
        foreach ($superficiesPorTemporada as $superficie) {
            $result[] = [
                'temporada' => $superficie->temporada,
                'superficie_total' => $superficie->superficie_total,
                'caduco' => $superficie->caduco,
                'pradera' => $superficie->pradera,
                'ciclo_corto' => $superficie->ciclo_corto,
                'persistente' => $superficie->persistente,
            ];
        }

        return response()->json($result);
    }

    public function getSuperficiePorTemporadaYCultivoOrganizacion(Request $request)
    {
        $organizacionId = $request->input('organizacion');
        $rangeStart = $request->input('rangeStart');
        $rangeEnd = $request->input('rangeEnd');

        // Obtener la tabla según la organización
        $table = SuperficieActiva::getCombinedSuperficieTemporadaYCultivoOrganization($organizacionId);

        // Obtener los canales asociados a la organización
        $canales = DB::table('canals')
            ->join('seccionrios', 'canals.seccionrio_id', '=', 'seccionrios.id')
            ->join('rios', 'seccionrios.rio_id', '=', 'rios.id')
            ->where('rios.organizacion_id', $organizacionId)
            ->pluck('canals.nombre');

        // Obtener la superficie total y por tipo de cultivo en el rango de temporadas seleccionado
        $superficiesPorTemporada = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->whereBetween('temporada', [$rangeStart, $rangeEnd])
            ->select(
                'temporada',
                DB::raw('SUM(area_ha) as superficie_total'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
                DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
            )
            ->groupBy('temporada')
            ->orderBy('temporada', 'asc')
            ->get();

        $result = [];
        foreach ($superficiesPorTemporada as $superficie) {
            $result[] = [
                'temporada' => $superficie->temporada,
                'superficie_total' => $superficie->superficie_total,
                'caduco' => $superficie->caduco,
                'persistente' => $superficie->persistente,
                'pradera' => $superficie->pradera,
                'ciclo_corto' => $superficie->ciclo_corto,
            ];
        }

        return response()->json($result);
    }

    public function getSuperficiePorTemporadaYCultivoRio(Request $request)
{
    $seccionrioId = $request->input('rio');
    $rangeStart = $request->input('rangeStart');
    $rangeEnd = $request->input('rangeEnd');

    // Obtener el río asociado a la sección del río y luego la organización a la que pertenece el río
    $rioId = DB::table('seccionrios')->where('id', $seccionrioId)->value('rio_id');
    $organizacionId = DB::table('rios')->where('id', $rioId)->value('organizacion_id');
    $table = SuperficieActiva::getCombinedSuperficieTemporadaYCultivoOrganization($organizacionId);

    // Obtener los canales asociados a la sección del río
    $canales = DB::table('canals')
        ->where('seccionrio_id', $seccionrioId)
        ->pluck('nombre');

    // Obtener la superficie total y por tipo de cultivo en el rango de temporadas seleccionado
    $superficiesPorTemporada = DB::table($table)
        ->whereIn('canal_v2', $canales)
        ->whereBetween('temporada', [$rangeStart, $rangeEnd])
        ->select(
            'temporada',
            DB::raw('SUM(area_ha) as superficie_total'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Caduco" THEN area_ha ELSE 0 END) as caduco'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Persistente" THEN area_ha ELSE 0 END) as persistente'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Pradera" THEN area_ha ELSE 0 END) as pradera'),
            DB::raw('SUM(CASE WHEN tipo_cultivo = "Ciclo corto" THEN area_ha ELSE 0 END) as ciclo_corto')
        )
        ->groupBy('temporada')
        ->orderBy('temporada', 'asc')
        ->get();

    $result = [];
    foreach ($superficiesPorTemporada as $superficie) {
        $result[] = [
            'temporada' => $superficie->temporada,
            'superficie_total' => $superficie->superficie_total,
            'caduco' => $superficie->caduco,
            'persistente' => $superficie->persistente,
            'pradera' => $superficie->pradera,
            'ciclo_corto' => $superficie->ciclo_corto,
        ];
    }

    return response()->json($result);
}


    public function getSuperficiePorOrganizacion(Request $request)
    {
        $organizacionId = $request->input('organizacion');

        // Obtener la tabla según la organización
        $table = SuperficieActiva::getCombinedSuperficieTemporadaYCultivoOrganization($organizacionId);

        // Obtener los canales asociados a la organización
        $canales = DB::table('canals')
            ->join('seccionrios', 'canals.seccionrio_id', '=', 'seccionrios.id')
            ->join('rios', 'seccionrios.rio_id', '=', 'rios.id')
            ->where('rios.organizacion_id', $organizacionId)
            ->pluck('canals.nombre');

        // Obtener las superficies agrupadas por temporada para los canales de la organización
        $superficies = DB::table($table)
            ->whereIn('canal_v2', $canales)
            ->groupBy('temporada')
            ->select(
                'temporada',
                DB::raw('SUM(area_ha) as area_ha')
            )
            ->get();

        // Calcular promedio, máxima y mínima superficie
        $promedio = $superficies->avg('area_ha');
        $maxima = $superficies->max('area_ha');
        $minima = $superficies->min('area_ha');

        // Encontrar las temporadas correspondientes a la superficie máxima y mínima
        $superficieMaxima = $superficies->firstWhere('area_ha', $maxima);
        $superficieMinima = $superficies->firstWhere('area_ha', $minima);

        return response()->json([
            'promedio' => $promedio,
            'maxima' => [
                'valor' => $maxima,
                'temporada' => $superficieMaxima->temporada,
            ],
            'minima' => [
                'valor' => $minima,
                'temporada' => $superficieMinima->temporada,
            ],
        ]);
    }
    public function getSuperficiePorRio(Request $request)
{
    $seccionrioId = $request->input('rio');

    // Obtener el río asociado a la sección del río y luego la organización a la que pertenece el río
    $rioId = DB::table('seccionrios')->where('id', $seccionrioId)->value('rio_id');
    $organizacionId = DB::table('rios')->where('id', $rioId)->value('organizacion_id');
    $table = SuperficieActiva::getCombinedSuperficieTemporadaYCultivoOrganization($organizacionId);

    // Obtener los canales asociados a la sección del río
    $canales = DB::table('canals')
        ->where('seccionrio_id', $seccionrioId)
        ->pluck('nombre');

    // Obtener las superficies agrupadas por temporada para los canales de la sección del río
    $superficies = DB::table($table)
        ->whereIn('canal_v2', $canales)
        ->groupBy('temporada')
        ->select(
            'temporada',
            DB::raw('SUM(area_ha) as area_ha')
        )
        ->get();

    // Calcular promedio, máxima y mínima superficie
    $promedio = $superficies->avg('area_ha');
    $maxima = $superficies->max('area_ha');
    $minima = $superficies->min('area_ha');

    // Encontrar las temporadas correspondientes a la superficie máxima y mínima
    $superficieMaxima = $superficies->firstWhere('area_ha', $maxima);
    $superficieMinima = $superficies->firstWhere('area_ha', $minima);

    return response()->json([
        'promedio' => $promedio,
        'maxima' => [
            'valor' => $maxima,
            'temporada' => $superficieMaxima->temporada,
        ],
        'minima' => [
            'valor' => $minima,
            'temporada' => $superficieMinima->temporada,
        ],
    ]);
}

}
